<?php

/**
 * @package    Sismos.Plugin
 * @subpackage Content.sismosautotoc
 *
 * @author     Martina Scholz <support@simplysmart-it.de>
 *
 * @copyright  (C) 2023 - 2025, SimplySmart-IT - Martina Scholz <https://simplysmart-it.de>
 * @license    GNU General Public License version 3 or later; see LICENSE
 * @link       https://simplysmart-it.de
 */

namespace Sismos\Plugin\Content\Sismosautotoc\Extension;

use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Event\Result\ResultAwareInterface;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Utility\Utility;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Sismosautotoc Content Plugin
 *
 * @since  1.0.0
 */
final class Sismosautotoc extends CMSPlugin implements SubscriberInterface
{
    /**
     * Affects constructor behavior. If true, language files will be loaded automatically.
     * Note this is only available in Joomla 3.1 and higher.
     * If you want to support 3.0 series you must override the constructor
     *
     * @var    boolean
     * @since  1.0.0
     */
    protected $autoloadLanguage = true;

    /**
     * The toc list with all headings.
     *
     * @var    array
     * @since  1.0.0
     */
    protected $tableToc = [];

    /**
     * The automatically generated TOC html string.
     *
     * @var    string
     * @since  1.0.0
     */
    protected $autotocHtml = "";

    /**
     * Check compatibility with pagebreak plugin and settings.
     *
     * @var    boolean
     * @since  1.0.0
     */
    protected $pageBreakCompatible;

    /**
     * The regex to search for headings.
     *
     * @var    string
     * @since  1.0.2
     */
    protected $searchTocRegex = '#(<\/?h(\d).*>)(.+)((?1))#iU';

    /**
     * The regex to search for pagebreaks.
     *
     * @var    string
     * @since  1.0.2
     */
    protected $pageBreakRegex = '#<hr(.*)class="system-pagebreak"(.*)\/?>#iU';

    /**
     * The article link.
     *
     * @var    string
     * @since  1.0.0
     */
    protected $articleLink;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   1.0.0
     *
     * @see libraries/src/Event/CoreEventAware.php
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onContentBeforeDisplay' => ['addAutoTOC', \Joomla\Event\Priority::MIN],
            'onContentPrepareForm'   => 'addAutoTOCOptionsToArticleForm',
            'onContentPrepare'       => ['createAutoTocFromArticle', \Joomla\Event\Priority::ABOVE_NORMAL],
        ];
    }

    /**
     * Prepare form and add my field.
     *
     * @param   \Joomla\Event\Event $event
     *
     * @return  void
     *
     * @since   1.0.0
     *
     */
    public function addAutoTOCOptionsToArticleForm(\Joomla\Event\Event $event)
    {
        /** @var Form $form  - The form to be altered*/

        [$form, $data] = array_values($event->getArguments());

        if (!$form instanceof Form) {
            return ;
        }

        $name = $form->getName();

        if ($name == 'com_content.article') {
            Form::addFormPath(\JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/forms');
            $form->loadFile('article-autotoc', false);
            if (\is_object($data) && property_exists($data, 'id') && !$data->id && $this->params->get('headline', '')) {
                $form->setFieldAttribute('autotoc_headline', 'default', $this->params->get('headline', 'PLG_CONTENT_SISMOSAUTOTOC_HEADLINE_DEFAULT'), 'attribs');
            }
        }

        return;
    }

    /**
     * This is the first stage in preparing content for output and is the
     * most common point for content orientated plugins to do their work.
     *
     * @param   \Joomla\Event\Event $event
     *
     * @return  void
     *
     * @since   1.0.0
     */
    public function createAutoTocFromArticle(\Joomla\Event\Event $event)
    {
        if (!$this->getApplication()->isClient('site')) {
            return;
        }

        /** @var string   $context  The context of the content being passed to the plugin */
        /** @var mixed    $row      An object with a "text" property */
        /** @var mixed    $params   Additional parameters. */

        [$context, $row, $params] = array_values($event->getArguments());

        if ($context !== 'com_content.article') {
            return;
        }

        if (!($params instanceof Registry) || !$params->get('autotoc_show', 0)) {
            return;
        }

        if (!\is_object($row)) {
            return;
        }

        if (!$this->checkPageBreak($row->introtext . $row->fulltext)) {
            return;
        }

        $this->articleLink = RouteHelper::getArticleRoute($row->slug, $row->catid, $row->language);

        $row->text = $this->generateTOC($row, $params);

        if ($event instanceof ResultAwareInterface) {
            $event->addResult($row);
            return;
        }
        // use GenericEvent approach
        $result   = $event->getArgument('result') ?: [];   // get the result argument from GenericEvent
        $result[] = $row;                                // add your return value into the array
        $event->setArgument('result', $result);          // write back the updated result into the GenericEvent instance


        return;
    }

    /**
     * This is a request for information that should be placed
     * immediately before the generated content.
     *
     * @param   \Joomla\Event\Event $event
     *
     * @return  void
     *
     * @since   1.0.0
     */
    public function addAutoTOC(\Joomla\Event\Event $event)
    {
        /** @var string   $context  The context of the content being passed to the plugin */
        /** @var mixed    $row      An object with a "text" property */
        /** @var mixed    $params   Additional parameters. */

        [$context, $row, $params] = array_values($event->getArguments());

        if (!$this->getApplication()->isClient('site')) {
            return;
        }

        if ($context !== 'com_content.article') {
            return;
        }

        if (!\is_object($row)) {
            return;
        }

        if (!($params instanceof Registry) || !$params->get('autotoc_show', 0)) {
            return;
        }

        if (!$this->pageBreakCompatible) {
            return;
        }

        if (!$this->autotocHtml) {
            return;
        }

        $row->toc = $this->autotocHtml;

        $event->setArgument(1, $row);

        return;
    }

    /**
     * Generate the TOC from the text.
     *
     * @param   object  $row     The row object.
     * @param   object  $params  The plugin parameters.
     *
     * @return  string  The text with the TOC.
     *
     * @since   1.0.0
     */
    private function generateTOC($row, $params): string
    {
        // Get the maximum level of headings to include.
        $maxLevel = (int) $params->get('autotoc_maxlevel', $this->params->get('heading_maxlevel', 3));

        // The text to generate the TOC from.
        $text = $row->text;

        // Find all headings in text and put in $matches.
        $matches = [];
        preg_match_all($this->searchTocRegex, $text, $matches, PREG_SET_ORDER);

        if (!empty(\count($matches))) {
            $text = preg_replace_callback(
                $this->searchTocRegex,
                function ($matches) {
                    $alias = ApplicationHelper::stringURLSafe($matches[3]);
                    return '<a id="' . $alias . '-' . $matches[2] . '"></a>' . $matches[0];
                },
                $text
            );
        }

        $splitPages       = [$text];
        $pageBreakMatches = [];

        if ($this->checkPageBreak($text)) {
            // Split the text into pages if necessary.
            $splitPages = preg_split($this->pageBreakRegex, $text);

            // Find all headings for pagebreaks if necessary.
            preg_match_all($this->pageBreakRegex, $text, $pageBreakMatches, PREG_SET_ORDER);
        }

        foreach ($splitPages as $index => $page) {
            // Find all headings in text and put in $matches.
            $matches = [];
            preg_match_all($this->searchTocRegex, $page, $matches, PREG_SET_ORDER);

            // Add the first page to TOC.
            if (!empty($pageBreakMatches) && $index === 0) {
                $this->tableToc = array_merge(
                    $this->tableToc,
                    [new Registry([
                        'link'  => $this->articleLink,
                        'title' => $row->title,
                        'level' => 0,
                    ])]
                );
            }

            // Create TOC from text.
            if (!empty($matches)) {
                $this->tableToc = array_merge($this->tableToc, $this->createToc($matches, $maxLevel, $index));
            }

            // Add pagebreaks to TOC.
            if (!empty($pageBreakMatches) && isset($pageBreakMatches[$index])) {
                $pageBreakAttributes = Utility::parseAttributes($pageBreakMatches[$index][1]);

                $title = Text::sprintf('PLG_CONTENT_SISMOSAUTOTOC_PAGEBREAK_PAGE_NUM', $index + 2);

                if (@$pageBreakAttributes['alt']) {
                    $title = stripslashes($pageBreakAttributes['alt']);
                } elseif (@$pageBreakAttributes['title']) {
                    $title = stripslashes($pageBreakAttributes['title']);
                }

                $this->tableToc = array_merge(
                    $this->tableToc,
                    [new Registry([
                        'link'  => $this->articleLink . '&limitstart=' . $index + 1,
                        'title' => $title,
                        'level' => 0,
                    ])]
                );
            }
        }

        // Render the TOC.
        if (\is_array($this->tableToc) && !empty($this->tableToc)) {
            $wa = $this->getApplication()->getDocument()->getWebAssetManager();
            $wa->registerAndUseStyle('sismosautotc', 'media/plg_content_sismosautotoc/css/sismosautoc.css');

            $headline = '';
            if ($params->get('autotoc_show_headline', 2) && $params->get('autotoc_show_headline', 2) < 2) {
                $headline = htmlspecialchars($params->get('autotoc_headline', 'PLG_CONTENT_SISMOSAUTOTOC_HEADLINE_DEFAULT'), ENT_QUOTES, 'UTF-8');
            } elseif ($params->get('autotoc_show_headline', 2) === 2 && $this->params->get('show_headline', 0)) {
                $headline = htmlspecialchars($this->params->get('headline', 'PLG_CONTENT_SISMOSAUTOTOC_HEADLINE_DEFAULT'), ENT_QUOTES, 'UTF-8');
            }

            $layoutPath = 'toc' . (($this->params->get('collapsable', 0)) ? '_collapse' : '');
            $path       = PluginHelper::getLayoutPath('content', 'sismosautotoc', $layoutPath);
            ob_start();
            include $path;
            $this->autotocHtml = ob_get_clean();
        }

        return $text;
    }

    /**
     * Create the TOC from the matches.
     *
     * @param   array  $matches   The matches from the regex.
     * @param   int    $maxLevel  The maximum level of headings to include.
     *
     * @return  array  The TOC.
     *
     * @since   1.0.0
     */
    private function createToc($matches, $maxLevel, $page = null): array
    {
        $tableToc = [];

        foreach ($matches as $heading) {
            if ($heading[0] && $heading[2] && (int) $heading[2] <= $maxLevel) {
                $attrs = Utility::parseAttributes($heading[1]);

                if ($heading[3]) {
                    $title = (\array_key_exists('data-alt', $attrs) && $attrs['data-alt']) ? $attrs['data-alt'] : stripslashes($heading[3]);
                    $alias = ApplicationHelper::stringURLSafe($heading[3]);
                } else {
                    continue;
                }

                $pagination = isset($page) && $page > 0 ? '&limitstart=' . $page : '';

                $tableToc[] = new Registry([
                    'link'  => $this->articleLink . $pagination . '#' . $alias . '-' . $heading[2],
                    'title' => $title,
                    'level' => $heading[2],
                ]);
            }
        }

        return $tableToc;
    }

    /**
     * Check compatibility with pagebreak plugin and settings.
     *
     * @param   string  $text  The text to check.
     *
     * @return  boolean  True if compatible, false if not.
     *
     * @since   1.0.0
     */
    private function checkPageBreak($text = ''): bool
    {
        if (!isset($this->pageBreakCompatible)) {
            if ($this->pageBreakCompatible = !PluginHelper::isEnabled('content', 'pagebreak')) {
                return $this->pageBreakCompatible;
            }

            $pageBreak        = PluginHelper::getPlugin('content', 'pagebreak');
            $pagebreak_params = new Registry($pageBreak->params);

            if ($this->pageBreakCompatible = !(bool) $pagebreak_params->get('multipage_toc', 0)) {
                return $this->pageBreakCompatible;
            }

            $isStylePages = $pagebreak_params->get('style', 'pages') === 'pages';

            if (!$this->pageBreakCompatible = ($isStylePages && $text)) { // TODO Check ???
                $this->pageBreakCompatible = !((bool) preg_match($this->pageBreakRegex, $text));
            }
        }
        return $this->pageBreakCompatible;
    }
}
