<?php

/**
 * @package    Sismos.Plugin
 * @subpackage Content.sismosautotoc
 *
 * @author     Martina Scholz <support@simplysmart-it.de>
 *
 * @copyright  (C) 2023 - 2025, SimplySmart-IT - Martina Scholz <https://simplysmart-it.de>
 * @license    GNU General Public License version 3 or later; see LICENSE
 * @link       https://simplysmart-it.de
 */

namespace Sismos\Plugin\Content\Sismosautotoc\Field;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

use Joomla\CMS\Form\FormField;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;

/**
 * Provides a modal content edit
 *
 * @since  1.1.0
 */
class ModalEditPluginField extends FormField
{
    /**
     * The form field type.
     *
     * @var    string
     * @since  1.1.0
     */
    protected $type = 'ModalEditPlugin';

    /**
     * Layout to render
     *
     * @var    string
     * @since  1.1.0
     */
    protected $layout = 'joomla.form.field.sismos.modal-edit-plugin';

    /**
     * Url for modal edit
     *
     * @var    string
     * @since  1.1.0
     */
    protected $url = '';

    /**
     * Checkin url for modal edit
     *
     * @var    string
     * @since  1.1.0
     */
    protected $checkin = '';

    /**
     * Titles for the edit modal
     *
     * @var    string
     * @since  1.1.0
     */
    protected $modalTitle = '';

    /**
     * Icons for the button
     *
     * @var    string
     * @since  1.1.0
     */
    protected $buttonIcon = 'icon-pen-square';

    /**
     * Text for the button
     *
     * @var    string
     * @since  1.1.0
     */
    protected $buttonText = 'JACTION_EDIT';

    /**
     * Cached data for layout rendering
     *
     * @var    array
     * @since  1.1.0
     */
    protected $layoutData = [];

    /**
     * Method to attach a Form object to the field.
     *
     * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the `<field>` tag for the form field object.
     * @param   mixed              $value    The form field value to validate.
     * @param   string             $group    The field name group control value.
     *
     * @return  boolean  True on success.
     *
     * @see     FormField::setup()
     * @since   1.1.0
     */
    public function setup(\SimpleXMLElement $element, $value, $group = null)
    {
        $result = parent::setup($element, $value, $group);

        if (!$result) {
            return $result;
        }

        // Prepare Url and title
        $plugin = PluginHelper::getPlugin((string) $this->element['group'], (string) $this->element['plugin']);
        if (!isset($plugin->id)) {
            return false;
        }
        $url    = Route::_('index.php?option=com_plugins&task=plugin.edit&extension_id=' . (int) $plugin->id . '&tmpl=component&layout=modal', false);
        // if (version_compare(JVERSION, '5.0.0', '<'))
        // {
        // $url = Route::_('index.php?option=com_plugins&task=plugin.edit&extension_id=' . (int) $plugin->id . '&layout=modal', false);
        // }
        $this->__set('url', (string) $url);
        $this->__set('title', (string) $this->element['title']);
        $this->__set('icon', (string) $this->element['icon']);
        $this->__set('button', (string) $this->element['button']);
        $this->__set('checkin', (string) Route::_('index.php?option=com_plugins&task=plugins.checkin&format=json&cid[]=' . (int) $plugin->id));

        return $result;
    }

    /**
     * Method to get certain otherwise inaccessible properties from the form field object.
     *
     * @param   string  $name  The property name for which to get the value.
     *
     * @return  mixed  The property value or null.
     *
     * @since   1.1.0
     */
    public function __get($name)
    {
        switch ($name) {
            case 'title':
                return $this->modalTitle;
            case 'icon':
                return $this->buttonIcon;
            case 'button':
                return $this->buttonText;
            default:
                return parent::__get($name);
        }
    }

    /**
     * Method to set certain otherwise inaccessible properties of the form field object.
     *
     * @param   string  $name   The property name for which to set the value.
     * @param   mixed   $value  The value of the property.
     *
     * @return  void
     *
     * @since   1.1.0
     */
    public function __set($name, $value)
    {
        switch ($name) {
            case 'title':
                $this->modalTitle = (string) $value;
                break;
            case 'icon':
                $this->buttonIcon = (string) $value;
                break;
            case 'button':
                $this->buttonText = (string) $value;
                break;
            case 'url':
                $this->url = (string) $value;
                break;
            case 'checkin':
                $this->checkin = (string) $value;
                break;
            default:
                parent::__set($name, $value);
        }
    }

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     *
     * @since   1.1.0
     */
    protected function getInput()
    {
        if (empty($this->layout)) {
            throw new \UnexpectedValueException(\sprintf('%s has no layout assigned.', $this->name));
        }

        // Get the layout data
        $data = $this->collectLayoutData();

        if (version_compare(JVERSION, '5.0.0', '<')) {
            $this->__set('layout', 'joomla.form.field.sismos.modal-edit-plugin-legacy');
        }

        return $this->getRenderer($this->layout)->render($data);
    }

    /**
     * Method to override layout paths
     *
     * @return  array
     *
     * @since   1.1.0
     */
    protected function getLayoutPaths()
    {
        $renderer = new FileLayout($this->layout);

        $renderer->setIncludePaths(parent::getLayoutPaths());

        $renderer->addIncludePaths(\JPATH_PLUGINS . '/content/sismosautotoc/layouts');

        $paths = $renderer->getIncludePaths();

        return $paths;
    }

    /**
     * Method to get the data to be passed to the layout for rendering.
     *
     * @return  array
     *
     * @since 1.1.0
     */
    protected function getLayoutData()
    {
        $data               = parent::getLayoutData();
        // ??? $data['canDo']      = $this->canDo;
        $data['url']        = $this->url;
        $data['modalTitle'] = $this->modalTitle;
        $data['buttonIcon'] = $this->buttonIcon;
        $data['buttonText'] = $this->buttonText;
        $data['checkin']    = $this->checkin;

        return $data;
    }

    /**
     * Method to get the data to be passed to the layout for rendering.
     * The data is cached in memory.
     *
     * @return  array
     *
     * @since 1.1.0
     *
     */
    protected function collectLayoutData(): array
    {
        if (method_exists(get_parent_class($this), 'collectLayoutData')) {
            return parent::collectLayoutData();
        }

        if ($this->layoutData) {
            return $this->layoutData;
        }

        $this->layoutData = $this->getLayoutData();

        return $this->layoutData;
    }
}
