<?php /**
* @package     Joomla.Plugins
* @subpackage  Content.Sismosautotoc
*
* @author      Martina Scholz <martina@simplysmart-it.de>
* @copyright   (C) 2023 Martina Scholz, SimplySmart-IT <https://simplysmart-it.de>
* @license     GNU General Public License version 3 or later; see LICENSE.txt
* @link        https://simplysmart-it.de
*/

namespace Sismos\Plugin\Content\Sismosautotoc\Extension;

use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Utility\Utility;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;
use Joomla\String\StringHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

final class Sismosautotoc extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 * Note this is only available in Joomla 3.1 and higher.
	 * If you want to support 3.0 series you must override the constructor
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * The toc list with all headings.
	 *
	 * @var    array
	 * @since  1.0.0
	 */
	protected $tableToc = [];

	/**
	 * The automatically generated TOC html string.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $autotocHtml = "";

	/**
	 * Check compatibility with pagebreak plugin and settings.
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $pageBreakCompatible;

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   4.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onContentBeforeDisplay' => ['addAutoTOC', \Joomla\Event\Priority::MIN],
			'onContentPrepareForm'   => 'addAutoTOCOptionsToArticleForm',
			'onContentPrepare'	     => ['createAutoTocFromArticle', \Joomla\Event\Priority::MIN],
		];
	}

	/**
	 * Prepare form and add my field.
	 *
	 * @param   \Joomla\Event\Event $event
	 *
	 * @return  void
	 *
	 * @since   3.9.0
	 */
	public function addAutoTOCOptionsToArticleForm(\Joomla\Event\Event $event)
	{
		/** @var Form $form  - The form to be altered*/

		[$form, $data] = $event->getArguments();

		if (!$form instanceof Form) {
			return ;
		}

		$name = $form->getName();

		if ($name =='com_content.article') {
			Form::addFormPath(dirname(__DIR__, 2) . '/forms');
			$form->loadFile('article-autotoc', false);
			if (is_object($data) && property_exists($data, 'id') && !$data->id) {
				$form->setFieldAttribute('autotoc_maxlevel', 'default', $this->params->get('heading_maxlevel', 3), 'attribs');
				$form->setFieldAttribute('autotoc_show_headline', 'default', $this->params->get('show_headline', 1), 'attribs');
				$form->setFieldAttribute('autotoc_headline', 'default', $this->params->get('headline', 0), 'attribs');
			}
		}

		return ;
	}

	/**
	 * This is the first stage in preparing content for output and is the
	 * most common point for content orientated plugins to do their work.
	 *
	 * @param   \Joomla\Event\Event $event
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function createAutoTocFromArticle(\Joomla\Event\Event $event)
	{
		if (!Factory::getApplication()->isClient('site')) {
			return;
		}

		[$context, $row, $params] = $event->getArguments();

		if ($context !== 'com_content.article') {
			return;
		}

		if (!$params || !$params->get('autotoc_show', 0)) {
			return;
		}

		if (is_object($row)) {
			if (!$this->checkPageBreake($row->introtext . $row->fulltext)) {
				return;
			}
			$this->generateTOC($row->text, $params);
		} else {
			if (!$this->checkPageBreake($row)) {
				return;
			}
			$this->generateTOC($row, $params);
		}

		$event->setArgument(1, $row);

		return;
	}

	/**
	 * This is a request for information that should be placed
	 * immediately before the generated content.
	 *
	 * @param   \Joomla\Event\Event $event
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function addAutoTOC(\Joomla\Event\Event $event)
	{
		[$context, $row, $params] = $event->getArguments();

		if (!\Joomla\CMS\Factory::getApplication()->isClient('site')) {
			return;
		}

		if ($context !== 'com_content.article') {
			return;
		}

		if (!is_object($row)) {
			return;
		}

		if (!$params || !$params->get('autotoc_show', 0)) {
			return;
		}

		if (!$this->pageBreakCompatible) {
			return;
		}

		if ($this->autotocHtml) :
			if (property_exists($row, 'toc')) {
				$row->toc .= $this->autotocHtml;
			} else {
				$row->toc = $this->autotocHtml;
			}
			$event->setArgument(1, $row);
		endif;
	}

	private function generateTOC(&$text, $params)
	{
		// Expression to search for.
		$regex = '#(<\/?h(\d).*>)(.+)((?1))#iU';

		// Find all headings in text and put in $matches.
		$matches = [];
		preg_match_all($regex, $text, $matches, PREG_SET_ORDER);

		if (!empty(count($matches))) {
			$text = preg_replace_callback(
				$regex,
				function ($matches) {
						$alias = ApplicationHelper::stringURLSafe($matches[3]);
						return '<a id="' . $alias . '-' . $matches[2] . '"></a>' . $matches[0];
				},
				$text
			);

			$maxLevel = (int) $params->get('autotoc_maxlevel', $this->params->get('heading_maxlevel', 3));
			$this->tableToc = $this->createToc($matches, $maxLevel);
			if (is_array($this->tableToc) && !empty($this->tableToc)) {
				$wa = Factory::getApplication()->getDocument()->getWebAssetManager();
				$wa->registerAndUseStyle('sismosautotc', 'media/plg_content_sismosautotoc/css/sismosautoc.css');

				$headline = '';
				if ($params->get('autotoc_show_headline', 2) && $params->get('autotoc_show_headline', 2) < 2) {
					$headline = $params->get('autotoc_headline', 'PLG_CONTENT_SISMOSAUTOTOC_HEADLINE_DEFAULT');
				} elseif ($params->get('autotoc_show_headline', 2) === 2 && $this->params->get('show_headline', 0)) {
					$headline = $this->params->get('headline', 'PLG_CONTENT_SISMOSAUTOTOC_HEADLINE_DEFAULT');
				}

				$layoutPath = 'toc' . (($this->params->get('collapsable', 0)) ? '_collapse' : '');
				$path = PluginHelper::getLayoutPath('content', 'sismosautotoc', $layoutPath);
				ob_start();
				include $path;
				$this->autotocHtml = ob_get_clean();
			}
		}

		return;
	}

	private function createToc($matches, $maxLevel)
	{
		$tableToc = [];

		foreach ($matches as $heading) {
			if ($heading[0] && $heading[2] && (int) $heading[2] <= $maxLevel) {
				$attrs = Utility::parseAttributes($heading[1]);

				if ($heading[3]) {
					$title = (array_key_exists('data-alt', $attrs) && $attrs['data-alt']) ?  $attrs['data-alt'] : stripslashes($heading[3]);
					$alias = ApplicationHelper::stringURLSafe($heading[3]);
				} else {
					continue;
				}

				$tableToc[] = new Registry([
					'link'   => '#' . $alias . '-' . $heading[2],
					'title'  => $title,
					'level'  => $heading[2]
				]);
			}
		}

		return $tableToc;
	}

	private function checkPageBreake($text = false)
	{
		if (!$this->pageBreakCompatible && $this->pageBreakCompatible !== false) {
			if ($this->pageBreakCompatible = !PluginHelper::isEnabled('content', 'pagebreak')) {
				return $this->pageBreakCompatible;
			}
			$pageBreak = PluginHelper::getPlugin('content', 'pagebreak');
			$pagebreak_params = new Registry($pageBreak->params);
			if (!$this->pageBreakCompatible = ($pagebreak_params->get('style', '') === 'pages') && $text) {
				$regex_pagebreak = '#<hr(.*)class="system-pagebreak"(.*)\/?>#iU';
				$this->pageBreakCompatible = !((bool) preg_match($regex_pagebreak, $text));
			}
		}
		return $this->pageBreakCompatible;
	}
}
