<?php

/**
 * @package     Joomla.Site
 * @subpackage  mod_sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Module\Sismosappointment\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\FormModel;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Item Model for an Appointment.
 *
 * @since  1.6
 */

class AppointmentModel extends FormModel
{
    /**
     * Maps events to plugin groups.
     *
     * @var    array
     * @since  3.6
     */
    protected $events_map = null;

    /**
     * Constructor
     *
     * @param   array                 $config       An array of configuration options (name, state, dbo, table_path, ignore_request).
     * @param   MVCFactoryInterface   $factory      The factory.
     * @param   FormFactoryInterface  $formFactory  The form factory.
     *
     * @since   3.6
     * @throws  \Exception
     */
    public function __construct($config = [], ?MVCFactoryInterface $factory = null, ?FormFactoryInterface $formFactory = null)
    {
        $config['events_map'] = $config['events_map'] ?? [];

        $this->events_map = array_merge(
            ['validate' => 'appointment'],
            $config['events_map']
        );

        parent::__construct($config, $factory);

        $this->setFormFactory($formFactory);
    }

    /**
     * Method to test whether a record can be deleted.
     *
     * @param   object  $record  A record object.
     *
     * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
     *
     * @since   1.6
     */
    protected function canDelete($record)
    {
        if (empty($record->id) || ($record->state != -2)) {
            return false;
        }

        // TODO also own edit contact
        return Factory::getUser()->authorise('core.edit', 'com_contact.contact.' . (int) $record->id);
    }

    /**
     * Method to get a single record.
     *
     * @param   integer  $pk  The id of the primary key.
     *
     * @return  mixed  Object on success, false on failure.
     */
    public function getItem($pk = null)
    {
        $pk    = (!empty($pk)) ? $pk : (int) $this->getState($this->getName() . '.id');
        $table = $this->getTable();

        if ($pk > 0) {
            // Attempt to load the row.
            $return = $table->load($pk);

            // Check for a table object error.
            if ($return === false) {
                // If there was no underlying error, then the false means there simply was not a row in the db for this $pk.
                if (!$table->getError()) {
                    $this->setError(Text::_('JLIB_APPLICATION_ERROR_NOT_EXIST'));
                } else {
                    $this->setError($table->getError());
                }

                return false;
            }
        }

        // Convert to the CMSObject before adding other data.
        $properties = $table->getProperties(1);
        $item       = ArrayHelper::toObject($properties, CMSObject::class);

        if (property_exists($item, 'params')) {
            $registry     = new Registry($item->params);
            $item->params = $registry->toArray();
        }

        return $item;
    }

    /**
     * Method for getting a form.
     *
     * @param   array    $data      Data for the form.
     * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
     *
     * @return  Form
     *
     * @since 1.0.5
     *
     * @throws \Exception
     */
    public function getForm($data = [], $loadData = false)
    {

        if (is_file(\dirname(__DIR__, 2) . '/forms/sismosappointmentform.xml')) {
            // Attempt to load the XML file.
            $xml  = simplexml_load_file(\dirname(__DIR__, 2) . '/forms/sismosappointmentform.xml');
            $form = $this->loadForm('mod_sismosappointment.form', $xml->asXML(), ['control' => 'jform', 'load_data' => $loadData]);
            if (empty($form)) {
                return false;
            }

            self::preprocessForm($form, []);

            return $form;
        }

        return false;
    }

    /**
     * Allows preprocessing of the Form object.
     *
     * @param   Form    $form   The form object
     * @param   mixed   $data   The data to be merged into the form object
     * @param   string  $group  The plugin group to be executed
     *
     * @return  void
     *
     * @see     FormField
     * @since 1.0.5
     * @throws  \Exception if there is an error in the form event.
     */
    protected function preprocessForm(Form $form, $data, $group = 'appointment') // TODO group ???
    {
        $form->setFieldAttribute('appointment_message', 'required', '', null);
        $form->setFieldAttribute('appointment_message', 'rows', '3', null);
        $form->setFieldAttribute('appointment_message', 'cols', '30', null);

        if (PluginHelper::isEnabled('content', 'confirmconsent')) {
            $lang = Factory::getApplication()->getLanguage();
            $lang->load('lib_sismos');
            $consentPlugin         = PluginHelper::getPlugin('content', 'confirmconsent');
            $consentPlugin->params = new Registry(json_decode($consentPlugin->params, true));
            // Get the consent box Text & the selected privacyarticle
            $consentboxText  = (string) $consentPlugin->params->get('consentbox_text', Text::_('LIB_SISMOS_FIELD_CONFIRMCONSENT_CONSENTNOTE_NOTE_DEFAULT'));
            $privacyArticle  = $consentPlugin->params->get('privacy_article', false);
            $privacyType     = $consentPlugin->params->get('privacy_type', 'article');
            $privacyMenuItem = $consentPlugin->params->get('privacy_menu_item', false);

            $form->load('
				<form>
					<fieldset name="default" addfieldprefix="Sismos\\Library\\Field">
						<field
							name="consentnote"
							type="SismosConsentNote"
							articleid="' . $privacyArticle . '"
							menu_item_id="' . $privacyMenuItem . '"
							privacy_type="' . $privacyType . '"
							label="LIB_SISMOS_FIELD_CONFIRMCONSENT_CONSENTNOTE_LABEL"
							>
							<option value="0">' . htmlspecialchars($consentboxText, ENT_COMPAT, 'UTF-8') . '</option>
						</field>
					</fieldset>
				</form>');
        }

        parent::preprocessForm($form, $data, $group);
    }

    /**
     * Method to validate the form data.
     *
     * @param   Form    $form   The form to validate against.
     * @param   array   $data   The data to validate.
     * @param   string  $group  The name of the field group to validate.
     *
     * @return  array|boolean  Array of filtered data if valid, false otherwise.
     *
     * @see     FormRule
     * @see     InputFilter
     * @since   1.6
     */
    public function validate($form, $data, $group = null)
    {
        // Include the plugins for the delete events.
        PluginHelper::importPlugin($this->events_map['validate']);

        $dispatcher = Factory::getContainer()->get('dispatcher');

        Factory::getApplication()->triggerEvent('onContentBeforeValidateData', [$form, &$data]);

        // Filter and validate the form data.
        $return = $form->process($data, $group);

        // Check for an error.
        if ($return instanceof \Exception) {
            $this->setError($return->getMessage());

            return false;
        }

        // Check the validation results.
        if ($return === false) {
            // Get the validation messages from the form.
            foreach ($form->getErrors() as $message) {
                $this->setError($message);
            }

            return false;
        }

        $data = $return;

        return $data;
    }
}
