<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\System\Sismosappointment\Helper;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Mail\Exception\MailDisabledException;
use Joomla\CMS\Mail\MailTemplate;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\String\PunycodeHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\User;
use Joomla\Database\ParameterType;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use PHPMailer\PHPMailer\Exception as phpMailerException;
use Sabre\VObject\Component\VCalendar;

class SismosappointmentHelper
{
    /**
     * The application instance
     *
     * @var    CMSApplicationInterface
     * @since  1.0
     */
    protected $app;

    /**
     * The contactId
     *
     * @var    int
     * @since  1.0
     */
    protected $contactId;

    /**
    * The The DatabaseDriver
    *
    * @var    DatabaseDriver|\JDatabaseDriver
    * @since  1.0
    */
    protected $db;

    /**
     * The Appointment Settings
     *
     * @var    stdClass|null
     * @since  1.0
     */
    protected $settings;

    /**
    * Method to instantiate the SismoappointmentHelper class object.
    *
    * @param   int                              $contactId        The contactId
    *
    * @since   1.0.0
    */
    public function __construct(int $contactId = 0)
    {
        $this->app       = Factory::getApplication();
        $this->db        = Factory::getDbo(); // Factory::getContainer()->get('DatabaseDriver');
        $this->contactId = $contactId;
        // Define the logger.
        Log::addLogger(['text_file' => 'plg_system_sismosappointment.php'], Log::ALL, ['plg_system_sismosappointment']);
    }

    /**
    * Method to set the ContactID.
    *
    * @param   int  $contactId        The contactId
    *
    * @since   1.0.0
    */
    public function setContactID(int $value)
    {
        $this->contactId = $value;
    }

    /**
     * Method to get the Appointment Settings if existent or from db as object.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0.3
     */
    // public static function getAppointmentSettings($contactId) {
    public function getAppointmentSettings()
    {
        if (\is_null($this->settings)) {
            $this->settings = $this->getSettings();
        }

        return $this->settings;
    }

    /**
     * Method to get the right integrated video conference solution.
     *
     * @return  array|boolean  the integration settings from appointment or from plugin settings.
     *
     * @since   1.0.3
     */
    protected function getIntegrationConferenceSolution()
    {
        if (!\is_array($this->getAppointmentSettings()->integration) || !\array_key_exists('ig_videoconf', $this->getAppointmentSettings()->integration)) {
            return false;
        }

        if ((string) $this->getAppointmentSettings()->integration['ig_videoconf'] === "-1") {
            if (!PluginHelper::isEnabled('system', 'sismosappointment')) {
                return false;
            }
            $plugin = PluginHelper::getPlugin('system', 'sismosappointment');

            if (!\is_object($plugin) || !$plugin->params) {
                return false;
            }

            $params = json_decode($plugin->params, true);

            if (!\array_key_exists('integration_videoconference', $params)) {
                return false;
            }

            $this->settings->integration['ig_global_videoconf'] = $params['integration_videoconference'];
        }

        if (!$this->getAppointmentSettings()->integration['ig_videoconf'] && !$this->getAppointmentSettings()->integration['ig_global_videoconf']) {
            return false;
        }

        return $this->getAppointmentSettings()->integration;
    }

    /**
     * Method to get the vCalendar settings.
     *
     * @param   string    $value        The value.
     *
     * @return  string  The setting as string
     *
     * @since   1.0.0
     */
    protected function getVCSettings($value)
    {

        if (\is_array($this->getAppointmentSettings()->vcalendar) && \array_key_exists($value, $this->getAppointmentSettings()->vcalendar) && $this->getAppointmentSettings()->vcalendar[$value]) {
            return $this->getAppointmentSettings()->vcalendar['$value'];
        }

        return 'PLG_SYSTEM_SISMOSAPPOINTMENT_BOOKING_VCALENDAR_' . strtoupper(str_replace('vc_', '', $value)) . '_DEFAULT';
    }

    /**
     * Method to get the Appointment Settings from db as object.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0.3
     */
    protected function getSettings()
    {
        $query = $this->db->getQuery(true)
            ->select(
                [
                    $this->db->quoteName('id'),
                    $this->db->quoteName('regular'),
                    $this->db->quoteName('duration'),
                    $this->db->quoteName('exclude'),
                    $this->db->quoteName('include'),
                    $this->db->quoteName('vcalendar'),
                    $this->db->quoteName('form'),
                    $this->db->quoteName('integration'),
                ]
            )
            ->from($this->db->quoteName('#__sismos_appointment_settings'))
            ->where($this->db->quoteName('contact_id') . ' = :contactid')
            ->bind(':contactid', $this->contactId, ParameterType::INTEGER);

        $this->db->setQuery($query);
        $results = $this->db->loadObject();

        if (!\is_null($results)) {
            $results->regular     = json_decode($results->regular, true);
            $results->duration    = json_decode($results->duration, true);
            $results->exclude     = json_decode($results->exclude, true);
            $results->include     = json_decode($results->include, true);
            $results->vcalendar   = json_decode($results->vcalendar, true);
            $results->form        = json_decode($results->form, true);
            $results->integration = json_decode($results->integration, true);
        }

        return $results;
    }

    /**
     * Method to get the Appointment Settings from db as object.
     *
     * @param  boolean  $formattedArray  Boolean if result should be formatted
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0
     */
    public function getAppointmentBookings($formattedArray = false)
    {
        $query = $this->db->getQuery(true)
            ->select(
                [
                    $this->db->quoteName('id'),
                    $this->db->quoteName('contact_id'),
                    $this->db->quoteName('name'),
                    $this->db->quoteName('appointment'),
                    $this->db->quoteName('offset'),
                    $this->db->quoteName('duration'),
                    $this->db->quoteName('email'),
                    $this->db->quoteName('message'),
                    $this->db->quoteName('block'),
                    $this->db->quoteName('created'),
                    $this->db->quoteName('meeting_url'),
                ]
            )
            ->from($this->db->quoteName('#__sismos_appointment_entries'))
            ->where($this->db->quoteName('contact_id') . ' = :contactid')
            ->bind(':contactid', $this->contactId, ParameterType::INTEGER)
            ->order($this->db->quoteName('appointment') . ' DESC');

        $this->db->setQuery($query);
        $results = $this->db->loadObjectList();

        if ($formattedArray) {
            $this->formattedBookingsArray($results);
        }

        return $results;
    }

    private function formattedBookingsArray(&$entries)
    {
        $entries = ArrayHelper::fromObject($entries);
        if (\is_array($entries)) {
            usort($entries, function ($a, $b) {
                return (Factory::getDate($a['appointment'], 'UTC') <=> Factory::getDate($b['appointment'], 'UTC'));
            });
        }
        $tz           = Factory::getApplication()->getIdentity()->getTimezone();
        $tzutc        = Factory::getDate('now', 'UTC');
        $tzoffsetuser = gmdate('H:i', floor(timezone_offset_get($tz, $tzutc)));
        foreach ($entries as &$entry) {
            // Convert a date to UTC based on the user timezone.
            if (\array_key_exists('appointment', $entry) && $entry['appointment'] != '') {
                // Get a date object based on the correct timezone.
                $utcDateTime = Factory::getDate($entry['appointment'], 'UTC');

                $utcDateTime->setTimezone($tz);

                // Transform the date string.
                $entry['appointment'] = $utcDateTime->format(Text::_('DATE_FORMAT_LC5'), true) . ' <small>UTC ' . (string) $tzoffsetuser . '</small>';
            }
            if (\array_key_exists('offset', $entry) && (int) $entry['offset'] !== 0) {
                $timezoneOffset  = $entry['offset'];
                $utcInfo         = ((int) $timezoneOffset > 0) ? '-' : '';
                $utcInfoTZOffset = ((float) $timezoneOffset < 0) ? ((float) $timezoneOffset * - 1) : (float) $timezoneOffset;
                $utcInfo .= gmdate('H:i', floor((float) ($utcInfoTZOffset * 60)));
                $entry['offset'] = 'UTC ' . $utcInfo . '</small>';
            } else {
                $entry['offset'] = 'UTC 00:00';
            }

            if (\array_key_exists('duration', $entry) && (int) $entry['duration'] !== 0) {
                $entry['duration'] = (string) gmdate('H:i', floor((float) ($entry['duration'] * 60)));
            }

            $created = Factory::getDate($entry['created'], 'UTC');
            $created->setTimezone($tz);
            $entry['created'] = HtmlHelper::date($created->format('Y-m-d H:i:s'), 'DATE_FORMAT_LC6');
        }
    }

    /**
     * Method to get the formatted Time String and UTC string for the user.
     *
     * @param  string  $utcDate  Date string in utc timezone
     * @param  int     $user_id  The user id (default 0)
     * @param  string  $format   The format for the Date string (default DATE_FORMAT_LC5)
     * @param  boolean  $local   The local boolean for the Date string
     *
     * @return  array  The appointment time string and UTC string as array.
     *
     * @since   1.0.0
     */
    private function getFormattedTimeUser($utcDate, $user_id = 0, $format = 'DATE_FORMAT_LC5', $local = true)
    {
        $data = [];
        $tz   = Factory::getApplication()->getIdentity()->getTimezone();
        if ($user_id) {
            $contact_user      = User::getInstance($user_id);
            $tz                = $contact_user->getTimezone();
        }
        $tzutc        = Factory::getDate('now', 'UTC');
        $tzoffsetuser = gmdate('H:i', floor(timezone_offset_get($tz, $tzutc)));

        // Get a date object based on the correct timezone.
        $utcDateTime = Factory::getDate($utcDate, 'UTC');

        $utcDateTime->setTimezone($tz);

        // Transform the date string.
        $data['datetime']  = $utcDateTime->format(Text::_($format), $local);
        $data['utcoffset'] = 'UTC ' . (string) $tzoffsetuser;

        return $data;
    }

    /**
     * Method to get the formatted Time String and UTC string for the guest.
     *
     * @param  string  $utcDate  Date string in utc timezone
     * @param  int     $offset   The offset from js or db
     * @param  string  $format   The format for the Date string (default DATE_FORMAT_LC5)
     *
     * @return  array  The appointment time string and UTC string as array.
     *
     * @since   1.0.0
     */
    private function getFormattedTimeUserGuest($utcDate, $offset = 0, $format = 'DATE_FORMAT_LC5')
    {
        $data = [];

        // Get a date object based on the correct timezone.
        $utcDateTime = Factory::getDate($utcDate, 'UTC');

        $intervalString = ((int) $offset < 0) ? 'PT' . ((int) ($offset) * -1) . 'M' : 'PT' . (int) ($offset) . 'M';
        $interval       = new \DateInterval($intervalString);

        if ($offset <= 0) {
            $utcDateTime->add($interval);
        } else {
            $utcDateTime->sub($interval);
        }

        // Transform the date string.
        $data['datetime'] = $utcDateTime->format(Text::_($format), true);

        $utcInfo         = ((int) $offset > 0) ? '-' : '+';
        $utcInfoTZOffset = ((float) $offset < 0) ? ((float) $offset * - 1) : (float) $offset;
        $utcInfo .= gmdate('H:i', floor((float) ($utcInfoTZOffset * 60)));
        $data['utcoffset'] = 'UTC ' . (string) $utcInfo;

        return $data;
    }

    /**
     * Method to get the Appointment from db by activation token as object.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0
     */
    protected function getAppointmentBookingByToken($token)
    {
        $query     = $this->db->getQuery(true);
        $query->select(
            [
                $this->db->quoteName('id'),
                $this->db->quoteName('contact_id'),
                $this->db->quoteName('name'),
                $this->db->quoteName('appointment'),
                $this->db->quoteName('offset'),
                $this->db->quoteName('duration'),
                $this->db->quoteName('email'),
                $this->db->quoteName('message'),
                $this->db->quoteName('activation'),
                $this->db->quoteName('block'),
                $this->db->quoteName('created'),
                $this->db->quoteName('meeting_url'),
            ]
        )
            ->from($this->db->quoteName('#__sismos_appointment_entries'))
            ->where($this->db->quoteName('activation') . ' = :token')
            ->where($this->db->quoteName('block') . ' = 1')
            ->bind(':token', $token, ParameterType::STRING);

        $this->db->setQuery($query);
        $results = $this->db->loadObject();

        return $results;
    }

    /**
     * Method to get the Appointment from db by md5 token as object.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0
     */
    protected function getAppointmentBookingBymd5($token)
    {
        $query     = $this->db->getQuery(true);
        $query->select(
            [
                $this->db->quoteName('id'),
                $this->db->quoteName('contact_id'),
                $this->db->quoteName('name'),
                $this->db->quoteName('appointment'),
                $this->db->quoteName('offset'),
                $this->db->quoteName('duration'),
                $this->db->quoteName('email'),
                $this->db->quoteName('message'),
                $this->db->quoteName('activation'),
                $this->db->quoteName('block'),
                $this->db->quoteName('created'),
                $this->db->quoteName('meeting_url'),
            ]
        )
            ->from($this->db->quoteName('#__sismos_appointment_entries'))
            ->where("md5(concat(" . $this->db->quoteName('email') . ",'-'," . $this->db->quoteName('appointment') . ")) = :token")
            ->bind(':token', $token, ParameterType::STRING);

        $this->db->setQuery($query);
        $results = $this->db->loadObject();

        return $results;
    }

    /**
     * Method to get the Appointment from db by id.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since   1.0
     */
    protected function getAppointmentBookingById($pk)
    {
        $query     = $this->db->getQuery(true);
        $query->select(
            [
                $this->db->quoteName('id'),
                $this->db->quoteName('contact_id'),
                $this->db->quoteName('name'),
                $this->db->quoteName('appointment'),
                $this->db->quoteName('offset'),
                $this->db->quoteName('duration'),
                $this->db->quoteName('email'),
                $this->db->quoteName('message'),
                $this->db->quoteName('activation'),
                $this->db->quoteName('block'),
                $this->db->quoteName('created'),
                $this->db->quoteName('meeting_url'),
            ]
        )
            ->from($this->db->quoteName('#__sismos_appointment_entries'))
            ->where($this->db->quoteName('id') . ' = :id')
            ->bind(':id', $pk, ParameterType::INTEGER);

        $this->db->setQuery($query);
        $results = $this->db->loadObject();

        return $results;
    }

    public function getSingleAppointment($pk)
    {
        $appointment = $this->getAppointmentBookingById($pk);
        $data        = ArrayHelper::fromObject($appointment);
        unset($data['activation']);
        $user = $this->app->getIdentity();
        $date = Factory::getDate($data['appointment'], 'UTC');
        $date->setTimezone($user->getTimezone());
        $data['appointment_datetime'] = $date->format('Y-m-d H:i', true, false);
        unset($data['appointment']);
        unset($data['contact_id']);
        $skip = ['appointment_datetime',''];
        foreach ($data as $key => $value) {
            if (!\in_array($key, $skip) && strpos('appointment_', $key) === false) {
                $data['appointment_' . $key] = $value;
                unset($data[$key]);
            }
        }
        return $data;
    }

    /**
     * Method to activate an Appointment (opt-in).
     *
     * @param   string  $token    random activation token
     *
     * @return  boolean|string  Result as boolean or redirect url
     *
     * @since   1.0.6
     */
    public function activateAppointmentBooking($token)
    {

        $entry =  $this->getAppointmentBookingByToken($token);

        if (\is_null($entry)) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_NOT_FOUND'), Log::ERROR);
            Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_NOT_FOUND'), false), 'info');
            return false;
        }

        $this->contactId = $entry->contact_id;

        if (!$this->setActiveAppointmentBooking((int) $entry->id)) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_ACTIVATION_FAILS'), Log::ERROR);
            Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_ACTIVATION_FAILS'), false), 'error');
            return false;
        }

        $data = ArrayHelper::fromObject($this->getAppointmentBookingById((int) $entry->id));

        // zoom and nextcloud integration
        $meeting_url = false;
        if ($conferenceSettings = $this->getIntegrationConferenceSolution()) {
            $pg_plugin   = (\array_key_exists('ig_global_videoconf', $conferenceSettings)) ? (string) $conferenceSettings['ig_global_videoconf'] : $this->getAppointmentSettings()->integration['ig_videoconf'];
            $meeting_url = (PluginHelper::isEnabled('sismosappointment', $pg_plugin)) ? $this->saveMeetingUrl($pg_plugin, $data) : false;
        }

        //SendActivation E-Mail
        $data['meeting_url'] = $meeting_url;
        if (!$this->sendEmail($data)) {
            Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_MAIL_FAILS'), false), 'warning');
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_MAIL_FAILS'), Log::WARNING);
        }

        Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_ACTIVATION_SUCCESS'), false), 'success');

        $redirect = '';

        if (\array_key_exists('fo_activ_redirect', $this->getAppointmentSettings()->form) && $this->getAppointmentSettings()->form['fo_activ_redirect']) {
            $redirect = ((int) $this->getAppointmentSettings()->form['fo_activ_redirect'] === 1) ?
                Route::_("index.php?Itemid=" . $this->getAppointmentSettings()->form['fo_activ_redirect_menu']) :
                (((int) $this->getAppointmentSettings()->form['fo_activ_redirect'] === 2) ? Route::_($this->getAppointmentSettings()->form['fo_activ_redirect_url']) : '');
        }

        return $redirect;
    }

    /**
     * Method to activate an Appointment (opt-in).
     *
     * @param   string  $token    random activation token
     *
     * @return  boolean|string  Result as boolean or redirect url
     *
     * @since   1.0.6
     */
    public function cancelAppointmentBooking($token)
    {

        $entry =  $this->getAppointmentBookingBymd5($token);

        if (\is_null($entry)) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_CANCEL_NOT_FOUND'), Log::ERROR);
            Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_CANCEL_NOT_FOUND'), false), 'info');
            return false;
        }

        $this->contactId = $entry->contact_id;

        if (!$this->setCancelAppointmentBooking((int) $entry->id)) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_CANCEL_FAILS'), Log::ERROR);
            Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_CANCEL_FAILS'), false), 'error');
            return false;
        }

        $data = ArrayHelper::fromObject($this->getAppointmentBookingById((int) $entry->id));

        if (!$this->sendCancelEmail($data)) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_MAIL_CANCEL_FAILS'), Log::WARNING);
        }

        Factory::getApplication()->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_CANCELATION_SUCCESS'), false), 'info');

        $redirect = '';

        if (\array_key_exists('fo_cancel_redirect', $this->getAppointmentSettings()->form) && $this->getAppointmentSettings()->form['fo_cancel_redirect']) {
            $redirect = ((int) $this->getAppointmentSettings()->form['fo_cancel_redirect'] === 1) ?
                Route::_("index.php?Itemid=" . $this->getAppointmentSettings()->form['fo_cancel_redirect_menu']) :
                (((int) $this->getAppointmentSettings()->form['fo_cancel_redirect'] === 2) ? Route::_($this->getAppointmentSettings()->form['fo_cancel_redirect_url']) : '');
        }

        return $redirect;
    }
    /**
     * Method to generate a meeting url via plugin and save it to db.
     *
     * @param   string    $pg_plugin   The clean plugin name.
     * @param   array     $entry       The entry data as array.
     *
     * @return  string|boolean  url on success false on failure.
     *
     * @since   1.0.3
     */
    private function saveMeetingUrl($pg_plugin, $entry)
    {
        PluginHelper::importPlugin('sismosappointment', $pg_plugin);
        $dispatcher = $this->app->getDispatcher();
        $user_id    = (\is_array($this->getAppointmentSettings()->integration) && \array_key_exists('igv_userid', $this->getAppointmentSettings()->integration) && $this->getAppointmentSettings()->integration['igv_userid']) ? $this->getAppointmentSettings()->integration['igv_userid'] : '';
        $password   = (\is_array($this->getAppointmentSettings()->integration) && \array_key_exists('igv_password', $this->getAppointmentSettings()->integration) && $this->getAppointmentSettings()->integration['igv_password']) ? $this->getAppointmentSettings()->integration['igv_password'] : '';
        $default    = (\is_array($this->getAppointmentSettings()->integration) && \array_key_exists('ig_videoconf', $this->getAppointmentSettings()->integration) && ((string) $this->getAppointmentSettings()->integration['ig_videoconf'] === "-1" || $this->getAppointmentSettings()->integration['ig_videoconf'] === $pg_plugin)) ? true : false;
        $vcsettings = [
            'vc_title'     => Text::_($this->getVCSettings('vc_title')),
            'vc_desc'      => Text::_($this->getVCSettings('vc_desc')),
            'igv_user'     => $user_id,
            'igv_password' => $password,
            'igv_default'  => $default,
        ];
        $event = new \Joomla\Event\Event('onSismosMeetingActivate', [$pg_plugin]);
        $event->addArgument('entry', array_merge($entry, $vcsettings));
        $result      = $dispatcher->dispatch('onSismosMeetingActivate', $event);
        $meeting_url = $result->getArgument('meeting_url', false);

        if ($meeting_url) {
            try {
                $query     = $this->db->getQuery(true);

                $query->update($this->db->quoteName('#__sismos_appointment_entries'))
                    ->set($this->db->quoteName('meeting_url') . '= :url')
                    ->where($this->db->quoteName('id') . '= :id')
                    ->bind(':url', $meeting_url, ParameterType::STRING)
                    ->bind(':id', $entry['id'], ParameterType::INTEGER);

                $this->db->setQuery($query);

                $this->db->execute();
            } catch (\Exception $e) {
                $this->log(\sprintf('Error save Meeting Url to DB: ', $e->getMessage()), Log::ERROR);
            }
        }

        return $meeting_url;
    }

    /**
     * Method to send an confirmation or activation email from modul.
     *
     * @param   array    $data   The data to send in the email.
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    public function sendEmail($data)
    {
        foreach ($data as $key => $value) {
            if ($key === 'appointment_datetime') {
                $data['appointment'] = $value;
                continue;
            }
            if ($key === 'contactId') {
                $data['contact_id'] = $value;
                $this->contactId    = (int) $value;
                continue;
            }
            if ($key === 'tzoffset') {
                $data['offset'] = $value;
                continue;
            }
            if (strpos($key, 'appointment_') !== false) {
                $data[str_replace('appointment_', '', $key)] = $value;
                continue;
            }
        }

        return $this->_sendEmail($data);
    }

    /**
     * Method to send an confirmation or activation email from ajax request.
     *
     * @param   int      $pk   The primary key.
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    public function sendEmailAjax($pk)
    {
        $data            = ArrayHelper::fromObject($this->getAppointmentBookingById($pk));
        $this->contactId = (int) $data['contact_id'];
        return $this->sendEmail($data);
    }

    /**
     * Method to send an confirmation or activation email.
     *
     * @param   array    $data   The data to send in the email.
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    private function sendMessage($data, $contact)
    {

        // Messaging if email fails

        $message = '';
        if ($data['message']) {
            $message = ($data['message']) ? $data['message'] : '';
        }

        if ($data['block'] == '-1') {
            $this->sendErrorMessage(
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_CANCEL_MESSAGE_SUBJECT', $contact->name),
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_CANCEL_MESSAGE_MESSAGE', $contact->name, 'Name: ' . $data['name'], $message)
            );
        } elseif ($data['block']) {
            $this->sendErrorMessage(
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ACTIVATION_MESSAGE_SUBJECT', $contact->name),
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ACTIVATION_MESSAGE_MESSAGE', $contact->name, 'Name: ' . $data['name'], $message)
            );
        } else {
            $this->sendErrorMessage(
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_CONFIRMATION_MESSAGE_SUBJECT', $contact->name),
                Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_CONFIRMATION_MESSAGE_MESSAGE', $contact->name, 'Name: ' . $data['name'], $message)
            );
        }
    }

    /**
     * Method to send an error message to Admin.
     *
     * @param   string    $subject   The subject for the message.
     * @param   string    $message   The message text
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    private function sendErrorMessage($subject, $message)
    {

        // Messaging to admin on Error

        // Push a notification to the site's super users, send email to user fails so the below message goes out
        /** @var MessageModel $messageModel */
        $messageModel = $this->app->bootComponent('com_messages')->getMVCFactory()->createModel('Message', 'Administrator');

        $messageModel->notifySuperUsers(
            $subject,
            $message
        );
    }

    /**
     * Method to send an confirmation or activation email.
     *
     * @param   array    $data   The data to send in the email.
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
    private function _sendEmail($data)
    {

        $model   = new \Joomla\Component\Contact\Site\Model\FormModel();
        $model   = $this->app->bootComponent('com_contact')->getMVCFactory()->createModel('form', 'Site', ['ignore_request' => true]);
        $contact = $model->getItem((int) $data['contact_id']);

        $settings = $this->getAppointmentSettings();

        if ($contact->email_to == '' && $contact->user_id != 0) {
            $contact_user      = User::getInstance($contact->user_id);
            $contact->email_to = $contact_user->get('email');
        }

        $appointment_user = $this->getFormattedTimeUser($data['appointment'], (int) $contact->user_id, 'DATE_FORMAT_LC2');

        $appointment_guest = $this->getFormattedTimeUserGuest($data['appointment'], (int) $data['offset'], 'DATE_FORMAT_LC2');

        $isBlocked = (\array_key_exists('block', $data) && $data['block']);

        $location = $this->getVCSettings('vc_location');

        $message = '';
        if ($data['message']) {
            $message = ($data['message']) ? Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_MESSAGE_LABEL', $data['message']) : '';
        }

        $body = '';
        if ($isBlocked && $settings->form['fo_mactivation']) {
            $body .= Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_LB_PLACEHOLDER_', $settings->form['fo_mactivation']);
        }

        $icsMeetingUrl = '';
        if (!$isBlocked && \array_key_exists('meeting_url', $data) && $data['meeting_url']) {
            $body .= Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_MEETINGURL_LABEL', $data['meeting_url']);
            $icsMeetingUrl = $data['meeting_url'];
        }

        if (!$isBlocked && $settings->form['fo_mconfirm']) {
            $body .= Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_LB_PLACEHOLDER_', $settings->form['fo_mconfirm']);
        }

        $templateData = [
            'sitename'    => $this->app->get('sitename'),
            'name'        => $data['name'],
            'contactname' => $contact->name,
            'email'       => PunycodeHelper::emailToPunycode($data['email']),
            'url'         => Uri::root(),
            'appointment' => HTMLHelper::date($data['appointment'], 'DATE_FORMAT_LC2'),
            'location'    => Text::_($location),
            'message'     => $message,
            'messageHTML' => ($message ? nl2br($message, false) : ''),
            'body'        => $body ?: '',
            'bodyHTML'    => ($body ? nl2br($body, false) : ''),
        ];

        $linkMode            = $this->app->get('force_ssl', 0) == 2 ? Route::TLS_FORCE : Route::TLS_IGNORE;
        $isSEFRewriteEnabled = $this->app->get('sef', 0) && $this->app->get('sef_rewrite', 0);
        if (!$isBlocked) {
            $siteFileName = str_replace(' ', '', File::makeSafe($templateData['sitename']));
            $icsContent   = $this->getVCalendarContent($data['appointment'], (int) $data['duration'], $data['name'], $icsMeetingUrl);
        } elseif ($isSEFRewriteEnabled) {
            $data['activate'] = Route::link(
                'site',
                Uri::root() . 'index.php/appointment.activate&token=' . $data['activation'],
                false,
                $linkMode,
                true
            );
        } else {
            $data['activate'] = Route::link(
                'site',
                Uri::root() . 'index.php?option=com_ajax&plugin=sismosappointment&format=json&task=appointment.activate&token=' . $data['activation'],
                false,
                $linkMode,
                true
            );
        }

        $cancelUrl = '';
        if ($settings->form['fo_cancel']) {
            $cancelUrl = $isSEFRewriteEnabled
            ? Route::link(
                'site',
                Uri::root() . 'index.php/appointment.cancel&token=' . md5($data['email'] . '-' . Factory::getDate($data['appointment'])->toSql()),
                false,
                $linkMode,
                true
            )
            : Route::link(
                'site',
                Uri::root() . 'index.php?option=com_ajax&plugin=sismosappointment&format=json&task=appointment.cancel&token=' . md5($data['email'] . '-' . Factory::getDate($data['appointment'])->toSql()),
                false,
                $linkMode,
                true
            );
        }

        $cancelLink = ($settings->form['fo_cancel']) ? Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_CANCELURL_LABEL', $cancelUrl) : '';

        $cancelLinkHtml = ($settings->form['fo_cancel']) ? Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_MAIL_CANCELURL_HTML_LABEL', $cancelUrl, $cancelUrl) : '';

        try {
            if ($isBlocked) {
                //Enquiry Mail to user
                $templateData['appointment'] = $appointment_user['datetime'] . ' (' . $appointment_user['utcoffset'] . ')';
                $templateData['url']         = Uri::root() . 'administrator/index.php?option=com_contact&task=contact.edit&id=' . $contact->id;

                $mailer = new MailTemplate('plg_system_sismosappointment.mail_activation_user', $this->app->getLanguage()->getTag());
                $mailer->addRecipient($contact->email_to);
                $mailer->setReplyTo($contact->email_to, $contact->name);
                $mailer->addTemplateData($templateData);
                $sent = $mailer->send();

                if (!$sent) {
                    $this->sendMessage($data, $contact);
                }

                //Activation Mail
                $templateData['activate']    = $data['activate'];
                $templateData['appointment'] = $appointment_guest['datetime'] . ' (' . $appointment_guest['utcoffset'] . ')';
                $templateData['url']         = Uri::root();
                $templateData['cancel']      = $cancelLink;
                $templateData['cancelHTML']  = $cancelLinkHtml;


                $mailer = new MailTemplate('plg_system_sismosappointment.mail_activation', $this->app->getLanguage()->getTag());
                $mailer->addRecipient($templateData['email']);
                $mailer->setReplyTo($contact->email_to, $contact->name);
                $mailer->addTemplateData($templateData);
                $sent = $mailer->send();
            } else {
                //Activation Mail to user
                $templateData['appointment'] = $appointment_user['datetime'] . ' (' . $appointment_user['utcoffset'] . ')';
                $templateData['url']         = Uri::root(); // . 'administrator/index.php?option=com_contact&task=contact.edit&id=' . $contact->id;

                $mailer = new MailTemplate('plg_system_sismosappointment.mail_user', $this->app->getLanguage()->getTag());
                $mailer->addRecipient($contact->email_to);
                $mailer->setReplyTo($contact->email_to, $contact->name);
                $mailer->addTemplateData($templateData);
                if (!$isBlocked) {
                    $mailer->addAttachment($siteFileName . '.ics', $icsContent);
                }
                $sent = $mailer->send();

                if (!$sent) {
                    $this->sendMessage($data, $contact);
                }

                //Confirmation Mail
                $templateData['appointment'] = $appointment_guest['datetime'] . ' (' . $appointment_guest['utcoffset'] . ')';
                $templateData['url']         = Uri::root();
                $templateData['cancel']      = $cancelLink;
                $templateData['cancelHTML']  = $cancelLinkHtml;

                $mailer = new MailTemplate('plg_system_sismosappointment.mail', $this->app->getLanguage()->getTag());
                $mailer->addRecipient($templateData['email']);
                $mailer->setReplyTo($contact->email_to, $contact->name);
                $mailer->addTemplateData($templateData);
                if (!$isBlocked) {
                    $mailer->addAttachment($siteFileName . '.ics', $icsContent);
                }
                $sent = $mailer->send();
            }
        } catch (MailDisabledException | phpMailerException $exception) {
            try {
                Log::add(Text::_($exception->getMessage()), Log::WARNING, 'jerror');

                $sent = false;
            } catch (\RuntimeException $exception) {
                Factory::getApplication()->enqueueMessage(Text::_($exception->getMessage()), 'warning');

                $sent = false;
            }
        }

        return $sent;
    }

    /**
     * Method to send an cancel info per email.
     *
     * @param   array    $data   The data to send in the email.
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0
     */
    private function sendCancelEmail($data)
    {

        $model   = new \Joomla\Component\Contact\Site\Model\FormModel();
        $model   = $this->app->bootComponent('com_contact')->getMVCFactory()->createModel('form', 'Site', ['ignore_request' => true]);
        $contact = $model->getItem((int) $data['contact_id']);

        if ($contact->email_to == '' && $contact->user_id != 0) {
            $contact_user      = User::getInstance($contact->user_id);
            $contact->email_to = $contact_user->get('email');
        }

        $appointment_user = $this->getFormattedTimeUser($data['appointment'], (int) $contact->user_id, 'DATE_FORMAT_LC2');

        $appointment_guest = $this->getFormattedTimeUserGuest($data['appointment'], (int) $data['offset'], 'DATE_FORMAT_LC2');

        $templateData = [
            'sitename'    => $this->app->get('sitename'),
            'name'        => $data['name'],
            'contactname' => $contact->name,
            'email'       => PunycodeHelper::emailToPunycode($data['email']),
            'url'         => Uri::root(),
            'appointment' => HTMLHelper::date($data['appointment'], 'DATE_FORMAT_LC2'),
        ];

        try {
            //Cancelation info Mail to user
            $templateData['appointment'] = $appointment_user['datetime'] . ' (' . $appointment_user['utcoffset'] . ')';
            $templateData['url']         = Uri::root() . 'administrator/index.php?option=com_contact&task=contact.edit&id=' . $contact->id;

            $mailer = new MailTemplate('plg_system_sismosappointment.mail_cancel_user', $this->app->getLanguage()->getTag());
            $mailer->addRecipient($contact->email_to);
            $mailer->setReplyTo($contact->email_to, $contact->name);
            $mailer->addTemplateData($templateData);
            $sent = $mailer->send();

            if (!$sent) {
                $this->sendMessage($data, $contact);
            }

            //Cancelation confirmation Mail
            $templateData['appointment'] = $appointment_guest['datetime'] . ' (' . $appointment_guest['utcoffset'] . ')';
            $templateData['url']         = Uri::root();

            $mailer = new MailTemplate('plg_system_sismosappointment.mail_cancel', $this->app->getLanguage()->getTag());
            $mailer->addRecipient($templateData['email']);
            $mailer->setReplyTo($contact->email_to, $contact->name);
            $mailer->addTemplateData($templateData);
            $sent = $mailer->send();
        } catch (MailDisabledException | phpMailerException $exception) {
            $this->log(Text::_($exception->getMessage()), Log::WARNING);
            $this->sendMessage($data, $contact);
            $sent = false;
        }

        return $sent;
    }

    /**
     * Method to get the vCalendar file data.
     *
     * @param   string    $startDate         The start datetime as string.
     * @param   int       $duration          The duration as integer in minutes.
     * @param   string    $participant_name  The name of the participant.
     * @param   string    $meeting_url       The optional meeting url.
     *
     * @return  string  The vcalandar file content as string
     *
     * @since   1.0.0
     */
    protected function getVCalendarContent($startDate, $duration, $participant_name, $meeting_url = '')
    {
        include_once JPATH_ROOT . "/plugins/system/sismosappointment/src/vendor/autoload.php";
        $sitename = $this->app->get('sitename');
        // ??? $settings = $this->getAppointmentSettings();
        $siteFileName  = str_replace(' ', '', File::makeSafe($sitename));
        $model         = $this->app->bootComponent('com_contact')->getMVCFactory()->createModel('form', 'Site', ['ignore_request' => true]);
        $contact       = $model->getItem((int) $this->contactId);
        $startDateTime = Factory::getDate($startDate, 'UTC');
        $endDateTime   = clone $startDateTime;
        $endDateTime->add(new \DateInterval('PT' . (int) $duration . 'M'));
        // TODO add Attendee emails???
        $vcalendar = new VCalendar([
            'VEVENT' => [
                'UID'         => HTMLHelper::date('now', 'ymdHis') . '@' . $siteFileName,
                'PRODID'      => 'SismosAppointments//Joomla@' . $sitename,
                'SUMMARY'     => Text::_($this->getVCSettings('vc_title')) . ' - ' . $sitename,
                'DTSTART'     => $startDateTime,
                'DTEND'       => $endDateTime,
                'LOCATION'    => Text::_($this->getVCSettings('vc_location')),
                'DESCRIPTION' => Text::_($this->getVCSettings('vc_desc')) . ' - ' . $sitename . ' \\n- ' . $contact->name . ' - ' . $participant_name . (($meeting_url) ? '\\n\\n' . $meeting_url : ''),
                'STATUS'      => 'CONFIRMED',

            ],
        ]);

        if ($contact->email_to) {
            $vcalendar->VEVENT->add(
                'ORGANIZER',
                $contact->email_to,
                [
                    'CN' => $contact->name,
                ]
            );
        }

        $icsContent = $vcalendar->serialize();

        return $icsContent;
    }

    /**
     * Method to get the Appointment from db by activation token as object.
     *
     * @return  boolean  Result as boolean
     *
     * @since   1.0
     */
    protected function setActiveAppointmentBooking($pk)
    {
        try {
            $query     = $this->db->getQuery(true);

            $query->update($this->db->quoteName('#__sismos_appointment_entries'))
                ->set($this->db->quoteName('block') . ' = 0')
                ->set($this->db->quoteName('activation') . ' = ""')
                ->where($this->db->quoteName('id') . ' = :id')
                ->bind(':id', $pk, ParameterType::INTEGER);

            $this->db->setQuery($query);

            $this->db->execute();
        } catch (\Exception $e) {
            $this->log(\sprintf('Error appointment activation save to DB: ', $e->getMessage()), Log::ERROR);
            return false;
        }
        return true;
    }

    /**
     * Method to cancel an Appointment by id.
     *
     * @return  boolean  Result as boolean
     *
     * @since   1.0
     */
    protected function setCancelAppointmentBooking($pk)
    {
        try {
            $query     = $this->db->getQuery(true);

            $query->update($this->db->quoteName('#__sismos_appointment_entries'))
                ->set($this->db->quoteName('block') . ' = -1')
                ->where($this->db->quoteName('id') . ' = :id')
                ->bind(':id', $pk, ParameterType::INTEGER);

            $this->db->setQuery($query);

            $this->db->execute();
        } catch (\Exception $e) {
            $this->log(\sprintf('Error appointment cancelation save to DB: ', $e->getMessage()), Log::ERROR);
            return false;
        }
        return true;
    }

    /**
     * Method to upate appointment data in DB .
     *
     * @param   array     $data    data appointment to update
     *
     * @return  boolean  Result as boolean
     *
     * @since   1.0
     */
    public function updateAppointment($data)
    {
        // can delete check permissions contact
        return $this->save($data);
    }

    protected function save($data)
    {
        if (!$data = $this->preprocessData($data)) {
            $this->log('Error preprocess data for save after edit appointment', Log::ERROR);
            return false;
            //TODO ErrorHandling
        }

        try {
            $pk        = $data['appointment_id'];
            $email     = ($data['appointment_email']) ? PunycodeHelper::emailToPunycode($data['appointment_email']) : '';
            $query     = $this->db->getQuery(true);
            $query->update($this->db->quoteName('#__sismos_appointment_entries'))
                ->set(
                    [
                        'name = :name',
                        'email = :email',
                        'message = :message',
                        'appointment = :appointment',
                        'duration = :duration',
                        'meeting_url = :meetingurl',
                    ]
                )
                    ->where($this->db->quoteName('id') . ' = :id')
                    ->bind(':name', $data['appointment_name'], ParameterType::STRING)
                    ->bind(':email', $email, ParameterType::STRING)
                    ->bind(':message', $data['appointment_message'], ParameterType::STRING)
                    ->bind(':appointment', $data['appointment_datetime'], ParameterType::STRING)
                    ->bind(':duration', $data['appointment_duration'], ParameterType::INTEGER)
                    ->bind(':meetingurl', $data['appointment_meeting_url'], ParameterType::STRING)
                    ->bind(':id', $pk, ParameterType::INTEGER);

            $this->db->setQuery($query);

            $this->db->execute();
        } catch (\Exception $e) {
            $this->log(\sprintf('Error save edited appointment data to DB: ', $e->getMessage()), Log::ERROR);
            return false;
        }

        return true;
    }

    protected function preprocessData($data)
    {
        $form = new Form('plg_sismosappointment.edit');

        FormHelper::addFormPath(\dirname(__DIR__, 2) . '/forms');
        $form->loadFile('sismosappointmententry');

        $form->setDatabase(Factory::getDbo());

        // TODO ??? validate Booking time
        $return = ($form->process($data));

        // Check for an error.
        if ($return instanceof \Exception) {
            $this->log(\sprintf('Error preprocess appointment data: ', $return->getMessage()), Log::ERROR);
            return false;
        }

        // Check the validation results.
        if ($return === false) {
            $listMessage = [];
            // Get the messages from messageQueue.
            $MessageQueue = $this->app->getMessageQueue();
            foreach ($MessageQueue as $msg) {
                $listMessage[] = $msg['message'];
            }

            // Get the validation messages from the form.
            if (!\count($listMessage)) {
                foreach ($form->getErrors() as $message) {
                    $listMessage[] = Text::_($message->getMessage());
                    /* Factory::getApplication()->enqueueMessage(
                        $message,
                        // Text::sprintf('JLIB_DATABASE_ERROR_FUNCTION_FAILED', $e->getCode(), $e->getMessage()),
                        'error'
                    ); */
                    // $this->setError($return->getMessage());
                }
            }
            $this->log(\sprintf('Error preprocess appointment data: ', implode('<br>', $listMessage)), Log::ERROR);
        }

        return $return;
    }

    /**
     * Method to delete an Appointment from db .
     *
     * @param   int     $pk    primary key
     *
     * @return  boolean  Result as boolean
     *
     * @since   1.0
     */
    public function deleteAppointmentByID($pk)
    {
        // can delete check permissions contact
        try {
            $query     = $this->db->getQuery(true);

            $query->delete($query->quoteName('#__sismos_appointment_entries'))
                    ->where($this->db->quoteName('id') . ' = :id')
                    ->bind(':id', $pk, ParameterType::INTEGER);

            $this->db->setQuery($query);

            $this->db->execute();
        } catch (\Exception $e) {
            $this->app->enque->setError($e->getMessage());
            $this->log(\sprintf('Error delete appointment: ', $e->getMessage()), Log::ERROR);

            return false;
        }
        return true;
    }

    /**
     * Method to delete Appointments from db .
     *
     * @param   string   $type    the type of delete Methode
     *
     * @return  boolean  Result as boolean
     *
     * @since   1.0
     */
    public function deleteAppointments($type)
    {
        // can delete check permissions contact
        try {
            $query     = $this->db->getQuery(true);

            $query->delete($query->quoteName('#__sismos_appointment_entries'));

            $date    = Factory::getDate('00:00:00', 'UTC');
            $dateStr = $date->toSql();

            switch ($type) {
                case 'O14D':
                    # code...
                    $older = clone $date;
                    $older->sub(new \DateInterval('P14D'));
                    $olderStr  = $older->toSql();
                    $contactId = $this->contactId;

                    $query->where($this->db->quoteName('contact_id') . ' = :contactid')
                        ->where('DATE(' . $this->db->quoteName('appointment') . ') < :expired OR DATE(' . $this->db->quoteName('created') . ') <= :older')
                        ->where('(' . $this->db->quoteName('block') . '=1 OR ' . $this->db->quoteName('block') . ' =-1)')
                        ->bind(':contactid', $contactId, ParameterType::INTEGER)
                        ->bind(':expired', $dateStr, ParameterType::STRING)
                        ->bind(':older', $olderStr, ParameterType::STRING);
                    break;

                case 'EXPIRED':
                    $contactId = $this->contactId;
                    $query->where($this->db->quoteName('contact_id') . ' = :contactid')
                        ->where('DATE(' . $this->db->quoteName('appointment') . ') < :expired')
                        ->bind(':contactid', $contactId, ParameterType::INTEGER)
                        ->bind(':expired', $dateStr, ParameterType::STRING);
                    break;

                default:
                    $this->app->enque->setError('No Delete Type selected!');

                    return false;

                    break;
            }

            $this->db->setQuery($query);

            $result = $this->db->execute();
        } catch (\Exception $e) {
            $this->app->enque->setError($e->getMessage());
            $this->log(\sprintf('Error delete multiple appointments: ', $e->getMessage()), Log::ERROR);

            return false;
        }
        return true;
    }

    public function exportCSV()
    {
        $date     = Factory::getDate('now', 'UTC');
        $filename = 'appointments_' . $date->format('Y-m-d_His_T');

        $csvDelimiter = ';';
        // TODO Settings Plugin $csvDelimiter = ComponentHelper::getComponent('com_actionlogs')->getParams()->get('csv_delimiter', ',');

        $this->app->setHeader('Content-Type', 'text/csv; charset=utf-8', true)
            ->setHeader('Content-disposition', 'attachment; filename="' . $filename . '.csv"', true)
            ->setHeader('Cache-Control', 'no-cache', true)
            ->sendHeaders();

        // Header row
        $rows[] = [
            'Id',
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_CSV_CONTACT_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_NAME_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_EMAIL_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_APPOINTMENT_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_CSV_OFFSETUSER_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_MESSAGE_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_OFFSETGUEST_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_FIELD_DURATION_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_CSV_ACTIVATED_LABEL'),
            'UTC',
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_CSV_CREATED_LABEL'),
            Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_CSV_MEETINGURL_LABEL'),
        ];

        $entries = $this->getAppointmentBookings();

        foreach ($entries as $entry) {
            $utcDateTime = Factory::getDate($entry->appointment, 'UTC');
            // Transform the date string.
            $userTime = $this->getFormattedTimeUser($utcDateTime, 0, 'DATE_FORMAT_LC5');
            // Transform the date string.
            $entry->appointment = $userTime['datetime'];
            // Translate enquiry offset into human readable string
            $enquiryTime     = $this->getFormattedTimeUserGuest($utcDateTime, (int) $entry->offset, 'DATE_FORMAT_LC5');
            $entry->duration = (string) gmdate('H:i', floor((float) ($entry->duration * 60)));
            $rows[]          = [
                $entry->id,
                $entry->contact_id,
                $entry->name,
                $entry->email,
                $entry->appointment,
                (string) $userTime['utcoffset'],
                empty($entry->message) ? ' ' : $entry->message,
                (string) $enquiryTime['utcoffset'],
                $entry->duration,
                ($entry->block) ? Text::_('JNO') : Text::_('JYES'),
                (string) $utcDateTime->format(Text::_('DATE_FORMAT_LC5'), false),
                $entry->created,
                $entry->meeting_url,
            ];
        }


        $output = fopen("php://output", "w");

        foreach ($rows as $row) {
            fputcsv($output, $row, $csvDelimiter);
        }

        fclose($output);

        $this->app->close();

        return true;
    }

    public function exportVCalendar($pk)
    {
        $date         = Factory::getDate('now', 'UTC');
        $siteFileName = str_replace(' ', '', File::makeSafe($this->app->get('sitename')));
        $filename     = 'appointment-' . $siteFileName . '_' . $date->format('Y-m-d_His_T');

        $this->app->setHeader('Content-Type', 'text/ics; charset=utf-8', true)
            ->setHeader('Content-disposition', 'attachment; filename="' . $filename . '.ics"', true)
            ->setHeader('Cache-Control', 'no-cache', true)
            ->sendHeaders();

        $appointment = $this->getAppointmentBookingById($pk);

        $icsMeetingUrl = (property_exists($appointment, 'meeting_url') && $appointment->meeting_url) ? $appointment->meeting_url : '';

        $icsContent = $this->getVCalendarContent($appointment->appointment, $appointment->duration, $appointment->name, $icsMeetingUrl);

        $output = fopen("php://output", "w");

        fputs($output, $icsContent);

        fclose($output);

        $this->app->close();

        return true;
    }

    /**
     * Log helper function
     *
     * @return  string
     */
    public function log($msg, $type)
    {
        $plugin    = PluginHelper::getPlugin('system', 'sismosappointment');
        $plgParams = new Registry($plugin->params);
        if ($plugin && $plgParams->get('log_on', 0)) {
            Log::add($msg, $type, 'plg_system_sismosappointment');
        }
    }
}
