<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\System\Sismosappointment\Field;

\defined('JPATH_BASE') or die;

use DateTime;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\Registry\Registry;

/**
 * Supports a time selection field.
 *
 * @since  1.0.8
 */
class TimeField extends FormField
{
    /**
     * The form field type.
     *
     * @var     string
     * @since   1.0.8
     */
    protected $type = 'time';

    /**
     * The filter.
     *
     * @var    integer
     * @since  1.0.8
     */
    protected $filter;

    /**
     * Method to attach a Form object to the field.
     *
     * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the `<field>` tag for the form field object.
     * @param   mixed              $value    The form field value to validate.
     * @param   string             $group    The field name group control value. This acts as an array container for the field.
     *                                      For example if the field has name="foo" and the group value is set to "bar" then the
     *                                      full field name would end up being "bar[foo]".
     *
     * @return  boolean  True on success.
     *
     * @see     FormField::setup()
     * @since   1.0.8
     */
    public function setup(\SimpleXMLElement $element, $value, $group = null)
    {
        $return = parent::setup($element, $value, $group);

        if ($return) {
            $this->filter       = (string) $this->element['filter'] ? (string) $this->element['filter'] : 'USER_UTC';
        }

        return $return;
    }

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     *
     * @since   1.0.8
     */
    protected function getInput()
    {

        // get relevant attributes which were defined in the XML form definition
        $attr = ' class="form-control ';
        $attr .= !empty($this->class) ? $this->class . '"' : '"';
        $attr .= !empty($this->element['min']) ? ' min="' . $this->element['min'] . '"' : '';
        $attr .= !empty($this->element['max']) ? ' max="' . $this->element['max'] . '"' : '';

        //Convert a date to local user time value.
        $user = Factory::getApplication()->getIdentity();

        // If a known filter is given use it.
        switch (strtoupper($this->filter)) {
            case 'SERVER_UTC':
                // Convert a date to UTC based on the server timezone.
                if ($this->value && $this->value != '') {
                    // Get a date object based on the correct timezone.
                    $date = Factory::getDate($this->value . ':00', 'UTC');
                    $date->setTimezone(new \DateTimeZone(Factory::getApplication()->get('offset')));

                    // Transform the date string.
                    $this->value = $date->format('H:i', true, false);
                }
                break;
            case 'USER_UTC':
                // Convert a date to UTC based on the user timezone.
                if ($this->value && $this->value != '') {
                    // Get a date object based on the correct timezone.
                    $date = Factory::getDate($this->value . ':00', 'UTC');
                    $date->setTimezone($user->getTimezone());

                    // Transform the date string.
                    $this->value = $date->format('H:i', true, false);
                }
                break;
        }

        // set up html, including the value and other attributes
        $html = '<input type="time" name="' . $this->name . '" value="' . $this->value . '"' . $attr . '/>';

        return $html;
    }

    /**
     * Method to filter a field value.
     *
     * @param   mixed     $value  The optional value to use as the default for the field.
     * @param   string    $group  The optional dot-separated form group path on which to find the field.
     * @param   Registry  $input  An optional Registry object with the entire data set to filter
     *                            against the entire form.
     *
     * @return  mixed   The filtered value.
     *
     * @since   1.0.8
     */
    public function filter($value, $group = null, ?Registry $input = null)
    {
        // Make sure there is a valid SimpleXMLElement.
        if (!($this->element instanceof \SimpleXMLElement)) {
            throw new \UnexpectedValueException(\sprintf('%s::filter `element` is not an instance of SimpleXMLElement', \get_class($this)));
        }
        if ((int) str_replace(':', '', $value) <= 0) {
            return '';
        }

        $datetime = new \DateTime();
        $value    = substr_count($value, ':') === 1 ? $value . ':00' : $value;
        $value    = \DateTime::createFromFormat('Y-m-d H:i:s', $datetime->format('Y-m-d') . ' ' . $value)->format('Y-m-d H:i:s');

        $app = Factory::getApplication();

        // Get the field filter type.
        $filter = (string) $this->filter;

        $return = $value;

        switch (strtoupper($filter)) {
            // Convert a date to UTC based on the server timezone offset.
            case 'SERVER_UTC':
                // Return an SQL formatted datetime string in UTC.
                $return = Factory::getDate($value, $app->get('offset'))->format('H:i');//->toSql();
                break;

                // Convert a date to UTC based on the user timezone offset.
            case 'USER_UTC':
                // Get the user timezone setting defaulting to the server timezone setting.
                $offset = $app->getIdentity()->getParam('timezone', $app->get('offset'));

                // Return an SQL formatted datetime string in UTC.
                $return = Factory::getDate($value, $offset)->format('H:i', false, false);//->toSql();
                break;
        }

        return $return;
    }
}
