<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\System\Sismosappointment\Extension;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Event\Content;
use Joomla\CMS\Event\ErrorEvent;
use Joomla\CMS\Event\Model;
use Joomla\CMS\Event\Plugin\AjaxEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;
use Joomla\String\StringHelper;
use SiSmOS\Plugin\System\Sismosappointment\Helper\SismosappointmentHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * SimplySmart-IT Appointment for contacts plugin.
 *
 * @package   Joomla.Plugin
 * @since     2.0.0
 */
class Sismosappointment extends CMSPlugin implements SubscriberInterface
{
    /**
     * Application object
     *
     * @var    CMSApplication
     * @since  1.0.0
     */
    protected $app;

    /**
     * Database object
     *
     * The database is injected by parent constructor
     *
     * @var    DatabaseDriver|\JDatabaseDriver
     * @since  1.0
     */
    protected $db;

    /**
     * Affects constructor behavior. If true, language files will be loaded automatically.
     *
     * @var    boolean
     * @since  1.0
     */
    protected $autoloadLanguage = true;


    /**
     * Module is loaded
     *
     * @var    boolean
     * @since  1.0
     */
    protected $moduleLoaded = false;

    /**
     * Redirect uri
     *
     * @var    string
     * @since  2.0.1
     */
    protected $redirectUri;


    /**
     * Returns an array of events this subscriber will listen to.
     *
     * The array keys are event names and the value can be:
     *
     *  - The method name to call (priority defaults to 0)
     *  - An array composed of the method name to call and the priority
     *
     * For instance:
     *
     *  * array('eventName' => 'methodName')
     *  * array('eventName' => array('methodName', $priority))
     *
     * @return  array
     *
     * @since   2.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onError'                 => 'onError',
            'onExtensionAfterSave'    => 'onExtensionAfterSave',
            'onAjaxSismosappointment' => 'onAjaxSismosappointment',
            'onContentPrepareData'    => 'onContentPrepareData',
            'onContentPrepareForm'    => 'onContentPrepareForm',
            'onContentAfterSave'      => 'onContentAfterSave',
            'onContentAfterDelete'    => 'onContentAfterDelete',
            'onContentAfterTitle'     => 'onContentAfterTitle',
            'onContentBeforeDisplay'  => 'onContentBeforeDisplay',
            'onContentAfterDisplay'   => 'onContentAfterDisplay',
        ];
    }

    /**
     * Constructor
     *
     * @param   DispatcherInterface  &$subject  The object to observe
     * @param   array                $config    An optional associative array of configuration settings.
     *                                          Recognized key values include 'name', 'group', 'params', 'language'
     *                                         (this list is not meant to be comprehensive).
     *
     * @since   1.0
     */
    public function __construct(&$subject, $config = [])
    {
        parent::__construct($subject, $config);

        // Define the logger.
        Log::addLogger(['text_file' => 'plg_system_sismosappointment.php'], Log::ALL, ['plg_system_sismosappointment']);
    }

    /**
     * Internal processor for all error handlers
     *
     * @param   ErrorEvent  $event  The event object
     *
     * @return  void
     *
     * @since   1.0
     *
     * @look public function onAfterInitialise()
     */
    public function onError(ErrorEvent $event)
    {
        /** @var \Joomla\CMS\Application\CMSApplication $app */
        $app = $event->getApplication();

        if ($app->isClient('administrator') || ((int) $event->getError()->getCode() !== 404)) {
            return;
        }

        $uri = Uri::getInstance();

        // These are the original URLs
        $orgQuery = rawurldecode($uri->toString(['path', 'query']));

        $this->log(Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_CATCH_URLQUERY', $orgQuery), Log::ERROR);

        if (StringHelper::strpos($orgQuery, '/oauth') !== false) {
            // TODO add nextcloud integration - no oauth recommended - no scopes yet???
            $queryParts = explode('?', $orgQuery);
            $pg_plugin  = substr($queryParts[0], strpos($queryParts[0], '/oauth') + \strlen('/oauth/'));
            if (PluginHelper::isEnabled('sismosappointment', $pg_plugin)) {
                $this->log(Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_OAUTH_STARTED', $pg_plugin), Log::INFO);
                PluginHelper::importPlugin('sismosappointment', $pg_plugin);
                $dispatcher = $this->app->getDispatcher();
                $newEvent   = new \Joomla\Event\Event('onCallbackIntegration', [$queryParts, $app]);
                $res        = $dispatcher->dispatch('onCallbackIntegration', $newEvent);
            }
            return;
        }

        if (StringHelper::strpos($orgQuery, 'appointment.activate') === false && StringHelper::strpos($orgQuery, 'appointment.cancel') === false) {
            return;
        }

        $queryParts = explode('&', $orgQuery);

        $queryToken = $queryParts[array_key_last($queryParts)];

        $this->redirectUri = Uri::base();

        if (StringHelper::strpos($queryToken, 'token=') === false) {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT'), Log::WARNING);
            $app->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT')), 'warning');
            $app->redirect($this->redirectUri, (int) 301);
            return;
        }

        if ($_SERVER['REQUEST_METHOD'] == 'HEAD') {
            $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_TOKEN_VERIFICATION_REQUESTMETHOD_HEAD'), Log::INFO);
            $app->redirect($this->redirectUri, (int) 301);
        }

        $queryToken = str_replace('token=', '', $queryToken);
        $queryToken = $this->db->escape($queryToken);

        if (StringHelper::strpos($orgQuery, 'appointment.activate') !== false) {
            $this->activateAppointment($queryToken);
        }

        if (StringHelper::strpos($orgQuery, 'appointment.cancel') !== false) {
            $this->cancelAppointment($queryToken);
        }

        $app->redirect($this->redirectUri, (int) 303);
    }

    /**
     * Activate an appointment using the provided token.
     *
     * @param   string  $token  The token to activate the appointment.
     *
     * @return  void
     *
     * @since   2.0.1
     */
    private function activateAppointment($token)
    {
        // Logic to activate appointment using the token
        $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_TOKEN_VERIFICATION_STARTED'), Log::INFO);
        if (($activation = $this->appointmentActivation($token)) !== false) {
            if (!$activation) {
                switch ((int) $this->params->get('activ_redirect', 0)) {
                    case 1:
                        $this->redirectUri = ($this->params->get('activ_redirect_menu', '')) ? Route::_("index.php?Itemid=" . $this->params->get('activ_redirect_menu')) : $this->redirectUri;
                        break;
                    case 2:
                        $this->redirectUri = ($this->params->get('activ_redirect_url', '')) ? Route::_($this->params->get('activ_redirect_url')) : $this->redirectUri;
                        break;
                    default:
                        break;
                }
            } else {
                $this->redirectUri = $activation;
            }
        };
        $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_TOKEN_VERIFICATION_END'), Log::INFO);
    }

    /**
     * Cancel an appointment using the provided token.
     *
     * @param   string  $token  The token to cancel the appointment.
     *
     * @return  void
     *
     * @since   2.0.1
     */
    private function cancelAppointment($token)
    {
        $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_TOKEN_CANCELATION_STARTED'), Log::INFO);
        if (($cancelation = $this->appointmentCancelation($token)) !== false) {
            if ($cancelation) {
                switch ((int) $this->params->get('cancel_redirect', 0)) {
                    case 1:
                        $this->redirectUri = ($this->params->get('cancel_redirect_menu', '')) ? Route::_("index.php?Itemid=" . $this->params->get('cancel_redirect_menu')) : $this->redirectUri;
                        break;
                    case 2:
                        $this->redirectUri = ($this->params->get('cancel_redirect_url', '')) ? Route::_($this->params->get('cancel_redirect_url')) : $this->redirectUri;
                        break;
                    default:
                        break;
                }
            } else {
                $this->redirectUri = $cancelation;
            }
        };
        $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_LOG_TOKEN_CANCELATION_END'), Log::INFO);
    }

    /**
     * Check the CSRF token for the current request.
     */
    private function checkToken()
    {
        $token = Session::getFormToken();
        // Check from header first
        if ($token === $this->app->input->server->get('HTTP_X_CSRF_TOKEN', '', 'alnum')) {
            return true;
        }

        // Then fallback to HTTP query
        if (!$this->app->getInput()->get($token, 0)) {
            return false;
        }

        return true;
    }

    /**
     * Generate a token for Meeting oAuth.
     * This method acts on table save, when a token doesn't already exist or a reset is required.
     *
     * @param   Model\SaveEvent  $event The onExtensionBeforeSave event.
     *
     * @return void
     *
     * @since 1.0
     */
    public function onExtensionAfterSave(Model\SaveEvent $event): void
    {
        /** @var Extension $table */
        $context   = $event->getContext();
        $table     = $event->getItem();
        $extension = $event->getData();

        if ($context !== 'com_plugins.plugin' || $table->folder !== 'sismosappointment') {
            return;
        }
        //get gentoken value and check
        if (Factory::getApplication()->getInput()->get('gentoken', null, 'int')) {
            // TODO ??? and nextcloud integration no oauth recommended - no scopes !!!
            $pg_plugin = str_replace('plg_sismosappointment_', '', $table->name);
            if (PluginHelper::isEnabled('sismosappointment', $pg_plugin)) {
                PluginHelper::importPlugin('sismosappointment', $pg_plugin);
                $dispatcher = $this->app->getDispatcher();
                $newEvent   = new \Joomla\Event\Event('onSismosGenToken', [$extension]);
                $res        = $dispatcher->dispatch('onSismosGenToken', $newEvent);
            }
        }
    }

    /**
     * Ajax call for SiSmOSAppointments.
     *
     * @param   AjaxEvent  $event  The event object
     *
     * @return  void
     *
     * @since 1.0
     */
    public function onAjaxSismosappointment(AjaxEvent $event)
    {
        if ($this->app->isClient('site')) {
            // Activation and Cancellation actions for appointments site

            $this->redirectUri = Uri::base();

            $token = $this->app->getInput()->get('token', '');

            if (empty($token)) {
                $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT'), Log::WARNING);
                $this->app->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT')), 'warning');
                $this->app->redirect($this->redirectUri, (int) 301);
            }

            switch ($this->app->getInput()->get('task', '')) {
                case 'appointment.activate':
                    $this->activateAppointment($token);
                    break;

                case 'appointment.cancel':
                    $this->cancelAppointment($token);
                    break;

                default:
                    $this->log(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT'), Log::WARNING);
                    $this->app->enqueueMessage(nl2br(Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_MESSAGE_TOKEN_LINK_INCORRECT')), 'warning');
                    $this->app->redirect($this->redirectUri, (int) 301);
                    break;
            }

            $this->app->redirect($this->redirectUri, (int) 303);
            $this->app->close();
            return;
        }

        $data = $this->app->getInput()->getData('jform');

        $result = ['reload' => false];

        if (!\is_array($data)) {
            echo new JsonResponse($result, Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NODATASEND_ERROR'), true);
            $this->app->close();
            return;
        }

        if (!$this->checkToken()) {
            echo new JsonResponse($result, Text::_('JLIB_ENVIRONMENT_SESSION_EXPIRED'), true);
            die('Invalid Token'); // ???
            return;
        }

        if (!$this->app->getInput()->get('task', '') || !\array_key_exists('id', $data)) {
            echo new JsonResponse($result, Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NOTASK_ERROR'), true);
            $this->app->close();
            return;
        }

        // TODO Test check access
        $user = $this->app->getIdentity();

        $contactEditable = $user->authorise('core.edit', 'com_contact') || ($user->authorise('core.edit.own', 'com_contact') && $data['jform[created_by]'] == $user->id);

        if (!$contactEditable) {
            echo new JsonResponse($result, Text::_('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_PERMISSION_ERROR'), true);
            $this->app->close();
            return;
        }

        $helper = new SismosappointmentHelper((int) $data['id']);

        switch ($this->app->getInput()->get('task', '')) {
            case 'CSVList':
                $helper->exportCSV();
                break;
            case 'ics':
                $pk = (int) $this->app->getInput()->get('sismosAptmaActionId', 0);
                if (!$pk) {
                    echo new JsonResponse($result, Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NOTFOUND_ERROR', $pk), true);
                    $this->app->close();
                    return;
                    break;
                }
                $helper->exportVCalendar($pk);
                break;
            case 'O14D':
                # code...
                $type             = $this->app->getInput()->get('task', '');
                $result['reload'] = $helper->deleteAppointments($type);
                break;

            case 'EXPIRED':
                $type             = $this->app->getInput()->get('task', '');
                $result['reload'] = $helper->deleteAppointments($type);
                break;

            case 'delSingle':
                $pk = (int) $this->app->getInput()->get('sismosAptmaActionId', 0);
                if (!$pk) {
                    echo new JsonResponse($result, Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NOTFOUND_ERROR', $pk), true);
                    $this->app->close();
                    return;
                    break;
                }

                $result['reload'] = $helper->deleteAppointmentByID($pk);
                break;
            case 'sendMail':
                $pk = (int) $this->app->getInput()->get('sismosAptmaActionId', 0);
                if (!$pk) {
                    echo new JsonResponse($result, Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NOTFOUND_ERROR', $pk), true);
                    $this->app->close();
                    return;
                    break;
                }

                $result['send'] = $helper->sendEmailAjax($pk);
                break;
            case 'loadEntryForm':
                $pk = (int) $this->app->getInput()->get('sismosAptmaActionId', 0);
                if (!$pk) {
                    echo new JsonResponse($result, Text::sprintf('PLG_SYSTEM_SISMOSAPPOINTMENT_ENTRIES_TABLE_NOTFOUND_ERROR', $pk), true);
                    $this->app->close();
                    return;
                    break;
                }
                $entry               = $helper->getSingleAppointment($pk);
                $result['entryData'] = $entry;
                break;
            case 'submitEntryForm':
                $res = $helper->updateAppointment([
                    'appointment_id'          => $this->app->getInput()->get('appointment_id', ''),
                    'appointment_datetime'    => $this->app->getInput()->getString('appointment_datetime', ''),
                    'appointment_duration'    => $this->app->getInput()->getString('appointment_duration', ''),
                    'appointment_name'        => $this->app->getInput()->getString('appointment_name', ''),
                    'appointment_email'       => $this->app->getInput()->getString('appointment_email', ''),
                    'appointment_message'     => $this->app->getInput()->getString('appointment_message', ''),
                    'appointment_meeting_url' => $this->app->getInput()->getString('appointment_meeting_url', ''),
                ]);
                // $res = false + msg ???
                $result['reload'] = $res;
                break;
            default:
                # code...
                break;
        }

        $result['ContactId'] = $data['id'];
        $result['Task']      = $this->app->getInput()->get('task', '');
        $html                = '';

        if ($result && !\in_array($this->app->getInput()->get('task', ''), ['CSVList', 'sendMail', 'ics', 'loadEntryForm'])) {
            $entries    = $helper->getAppointmentBookings(true);
            $layoutfile = new FileLayout('joomla.form.field.appointmententries.default.items', \JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/layouts');
            $html       = $layoutfile->render(['entries' => $entries]);
        }

        $result['FieldEntries'] = $html;

        $event->updateEventResult($result);
    }

    /**
     * Runs on content preparation
     *
     * @param   PrepareDataEvent  $event  The event
     *
     * @since   2.0.0
     *
     */
    public function onContentPrepareData(Model\PrepareDataEvent $event)
    {

        $context = $event->getContext();
        $data    = $event->getData();

        // Run this in frontend only
        if ($this->app->isClient('site')) {
            return true; // TODO FE edit
        }

        if (!\in_array($context, ['com_contact.contact'])) {
            return true;
        }

        if (\is_object($data)) {
            $contactId = $data->id ?? 0;

            // if (!isset($data->profile) && $contactId > 0) {
            if (!isset($data->sismosapmt) && $contactId > 0) {
                // Load the profile data from the database.
                $helper  = new SismosappointmentHelper($contactId);
                $results = $helper->getAppointmentSettings();

                // Add the appointment settings data.
                if (!\is_null($results)) {
                    $data->sismosapmt['id']                = $results->id;
                    $data->sismosapmt['sapmt-regular']     = $results->regular;
                    $data->sismosapmt['sapmt-duration']    = $results->duration;
                    $data->sismosapmt['sapmt-exclude']     = $results->exclude;
                    $data->sismosapmt['sapmt-include']     = $results->include;
                    $data->sismosapmt['sapmt-vcalendar']   = $results->vcalendar;
                    $data->sismosapmt['sapmt-form']        = $results->form;
                    $data->sismosapmt['sapmt-integration'] = $results->integration;
                }

                $entries = $helper->getAppointmentBookings();
                if (!\is_null($entries)) {
                    $data->sismosapmtentries['sapmt-entries'] = json_encode($entries);
                }

                $event->updateData($data);
            }
        }
    }

    /**
     * Add fields for the appointment data to the form
     *
     * @param   Model\PrepareFormEvent  $event  The event
     * @return  void
     *
     * @since  2.0.0
     */
    public function onContentPrepareForm(Model\PrepareFormEvent $event): bool
    {
        /* @var \Joomla\CMS\Form\Form $form */
        $form = $event->getForm();

        $client = $this->app->getName();

        if ($client === 'site') {
            return true;
        }

        $option = $this->app->input->get('option');

        if ($option != 'com_contact') {
            return true;
        }

        // Backend
        $form::addFormPath(JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/forms');
        $form->loadFile('sismosappointment', false);

        return true;
    }

    /**
     * The save event.
     *
     * @param   Model\AfterSaveEvent $event
     *
     * @return  void
     *
     * @since   2.0.0
     */
    public function onContentAfterSave(Model\AfterSaveEvent $event) //($context, $contact, $isNew): void
    {
        $context   = $event->getContext();
        $contact   = $event->getItem();

        if (!\in_array($context, ['com_contact.contact'])) {
            return;
        }

        $contactId = (int) $contact->id;

        $jformData = $this->app->input->get('jform', [], 'array');

        $data = [];

        $form = new Form('appointments');
        $form::addFormPath(__DIR__ . '/forms');
        $form->loadFile('sismosappointment');

        $formData = $form->filter($jformData);

        $data['contactId'] = $this->db->qn($contactId);

        if ($isValid = $form->validate($formData)) {
            foreach ($formData['sismosapmt'] as $key => $v) {
                if ($key === 'id') {
                    $data['id'] = $v;
                    continue;
                };
                if (!\is_array($v)) {
                    $v = [];
                }
                $data[str_replace('sapmt-', '', $key)] = json_encode($v);
            }

            // Create the base insert statement.
            $query = $this->db->getQuery(true);
            if ($data['id'] > 0) {
                $query->clear()
                ->update($this->db->quoteName('#__sismos_appointment_settings'))
                ->set([
                    $this->db->quoteName('regular') . ' = :regular',
                    $this->db->quoteName('duration') . ' = :duration',
                    $this->db->quoteName('exclude') . ' = :exclude',
                    $this->db->quoteName('include') . ' = :include',
                    $this->db->quoteName('vcalendar') . ' = :vcalendar',
                    $this->db->quoteName('form') . ' = :form',
                    $this->db->quoteName('integration') . ' = :integration',
                ])
                ->where($this->db->quoteName('id') . ' = :id')
                ->bind(':regular', $data['regular'], ParameterType::STRING)
                ->bind(':duration', $data['duration'], ParameterType::STRING)
                ->bind(':exclude', $data['exclude'], ParameterType::STRING)
                ->bind(':include', $data['include'], ParameterType::STRING)
                ->bind(':vcalendar', $data['vcalendar'], ParameterType::STRING)
                ->bind(':form', $data['form'], ParameterType::STRING)
                ->bind(':integration', $data['integration'], ParameterType::STRING)
                ->bind(':id', $data['id'], ParameterType::INTEGER);
            } else {
                $query->clear()
                    ->insert($this->db->quoteName('#__sismos_appointment_settings'))
                    ->columns(
                        [
                            $this->db->qn('contact_id'),
                            $this->db->qn('regular'),
                            $this->db->qn('duration'),
                            $this->db->qn('exclude'),
                            $this->db->qn('include'),
                            $this->db->qn('vcalendar'),
                            $this->db->qn('form'),
                            $this->db->qn('integration'),
                        ]
                    );
                $query->values(
                    implode(
                        ',',
                        $query->bindArray(
                            [$contactId, $data['regular'], $data['duration'],$data['exclude'],$data['include'],$data['vcalendar'],$data['form'],$data['integration']],
                            [ParameterType::INTEGER, ParameterType::STRING, ParameterType::STRING, ParameterType::STRING, ParameterType::STRING, ParameterType::STRING, ParameterType::STRING,ParameterType::STRING]
                        )
                    )
                );
            }

            // Set the query and execute the insert.
            $this->db->setQuery($query);

            try {
                $this->db->execute();
            } catch (\RuntimeException $e) {
                $this->app->enqueueMessage('<b>Appointments Error</b> - ' . $e->getMessage(), 'danger');
                // TODO $this->setError($e->getMessage());
                return;
            }
        }

        if ($isValid === false) {
            foreach ($form->getErrors() as $err) {
                $this->app->enqueueMessage('<b>Appointments Error</b> - ' . $err->getMessage(), 'danger');
            }
        }

        return;
    }

    /**
     * The after Delete event.
     *
     * @param   Object    $event  The event
     *
     * @return  void
     *
     * @since   2.0.0
     */
    public function onContentAfterDelete(Model\AfterDeleteEvent $event) //($context, $contact)
    {
        $context = $event->getContext();

        if (!\in_array($context, ['com_contact.contact'])) {
            return;
        }

        // Check if there is an id
        if (empty($event->getItem()->id)) {
            return '';
        }

        $contactId = $event->getItem()->id;

        // delete all assigned appointmentsettings
        $query = $this->db->getQuery(true);
        $query->delete('#__sismos_appointment_settings')
            ->where($this->db->quoteName('contact_id') . ' = :contactid')
            ->bind(':contactid', $contactId, ParameterType::INTEGER);

        $this->db->setQuery($query);
        $this->db->execute();

        // delete all assigned appointment_entries
        $query = $this->db->getQuery(true);
        $query->delete('#__sismos_appointment_entries')
            ->where($this->db->quoteName('contact_id') . ' = :contactid')
            ->bind(':contactid', $contactId, ParameterType::INTEGER);

        $this->db->setQuery($query);
        $this->db->execute();
    }

    /**
     * The display event.
     *
     * @param   Content\AfterTitleEvent $event
     *
     * @return  void
     *
     * @since   2.0.0
     */
    public function onContentAfterTitle(Content\AfterTitleEvent $event)
    {
        $result = $this->display($event->getContext(), $event->getItem(), $event->getParams(), [1,4]);

        if ($result) {
            $event->addResult($result);
        }
    }

    /**
     * The display event.
     *
     * @param   Content\BeforeDisplayEvent $event
     *
     * @return  void
     *
     * @since   2.0.0
     */
    public function onContentBeforeDisplay(Content\BeforeDisplayEvent $event)
    {
        $result = $this->display($event->getContext(), $event->getItem(), $event->getParams(), [2]);

        if ($result) {
            $event->addResult($result);
        }
    }

    /**
     * The display event.
     *
     * @param   Content\AfterDisplayEvent $event
     *
     * @return  void
     *
     * @since   2.0.0
     */
    public function onContentAfterDisplay(Content\AfterDisplayEvent $event)
    {
        $result = $this->display($event->getContext(), $event->getItem(), $event->getParams(), [3,4]);

        if ($result) {
            $event->addResult($result);
        }
    }

    /**
     * Performs the display event.
     *
     * @param   string    $context      The context
     * @param   stdClass  $item         The item
     * @param   Registry  $params       The params
     * @param   array     $displayType  The type
     *
     * @return  string
     *
     * @since   1.0
     */
    private function display($context, $item, $params, $displayType)
    {

        // Run this in frontend only
        if ($this->app->isClient('administrator')) {
            return;
        }

        if (!\in_array($context, ['com_contact.contact','com_contact.category'])) {
            return;
        }

        $sampt_enabled = $item->params->get('sapmt-enabled', 0);

        if (!$sampt_enabled) {
            return;
        }

        $display = ($context === 'com_contact.contact') ? $this->params->get('display', 0) : $this->params->get('displaylist', 0);

        if (!\in_array($display, $displayType)) {
            return;
        }

        $buttonText = ($context === 'com_contact.category') ? $this->params->get('displaylist_btntext', '') : $this->params->get('display_btntext', '');

        $module            = ModuleHelper::getModule('mod_sismosappointment', $context);
        $module->showtitle = 0;
        $modParams         = new Registry();
        $modParams->loadString('{"contactid":"' . $item->id . '","modal_title":"","modal_button":"' . $buttonText . '","modal_duration":"-1","modal_prep":"-1","modal_bstart":"-1","layout":"_:default","moduleclass_sfx":"","owncache":0,"cache_time":0,"module_tag":"div","bootstrap_size":"0","header_tag":"h3","header_class":"","style":"0"}');
        $module->params = $modParams;

        return ModuleHelper::renderModule($module, $modParams->toArray());
    }

    /**
     * Performs the appointment activation (opt in)
     *
     * @param   string    $token        The token
     *
     * @return  boolean|string
     *
     * @since   1.0
     */
    private function appointmentActivation($token)
    {
        //Activate
        $helper = new SismosappointmentHelper();

        return $helper->activateAppointmentBooking((string) $token);
    }

    /**
     * Performs the appointment cancelation
     *
     * @param   string    $token        The token
     *
     * @return  boolean|string
     *
     * @since   1.0
     */
    private function appointmentCancelation($token)
    {
        //Activate
        $helper = new SismosappointmentHelper();

        return $helper->cancelAppointmentBooking((string) $token);
    }

    /**
     * Log helper function
     *
     * @return  string
     */
    public function log($msg, $type)
    {
        if ($this->params->get('log_on', 1)) {
            Log::add($msg, $type, 'plg_system_sismosappointment');
        }
    }
}
