<?php

/**
 * @package     Joomla.Site
 * @subpackage  mod_sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Module\Sismosappointment\Site\Helper;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

use DateTime;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\OutputController;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Session\Session;
use Joomla\CMS\User\UserHelper;
use Joomla\Database\ParameterType;
use Joomla\Http\HttpFactory;
use Joomla\Registry\Registry;
use Laminas\Diactoros\Request;
use Laminas\Diactoros\Stream;
use Sabre\VObject\Reader;
use SiSmOS\Module\Sismosappointment\Site\Model\AppointmentModel;
use SiSmOS\Module\Sismosappointment\Site\Model\AppointmentsModel;

class RegularMode
{
    public const ALLWORKDAYS = -1;
    public const MONDAY      = 1;
    public const TUESDAY     = 2;
    public const WEDNESDAY   = 3;
    public const THURSDAY    = 4;
    public const FRIDAY      = 5;
    public const SATURDAY    = 6;
    public const SUNDAY      = 0;


    public const REGULARTIMES = [
        -1 => 'ALLWORKDAYS',
        1  => 'MONDAY',
        2  => 'TUESDAY',
        3  => 'WEDNESDAY',
        4  => 'THURSDAY',
        5  => 'FRIDAY',
        6  => 'SATURDAY',
        0  => 'SUNDAY',
    ];
}

class SismosappointmentHelper
{
    /**
     * The application instance
     *
     * @var    CMSApplicationInterface
     * @since 1.0.5
     */
    protected $app;

    /**
     * The formData
     *
     * @var    stdClass
     * @since  1.0
     */
    protected $formData;

    /**
     * Affects constructor behavior. If true, language files will be loaded automatically.
     *
     * @var    boolean
     * @since  1.0
     */
    protected $autoloadLanguage = true;

    /**
     * The contactId
     *
     * @var    int
     * @since  1.0
     */
    protected $contactId;

    /**
     * The moduleId
     *
     * @var    int
     * @since  1.0
     */
    protected $moduleId = -1;

    /**
     * The activeMenuId
     *
     * @var    int
     * @since  1.0
     */
    protected $activeMenuId = -1;

    /**
     * Is enabled for contactId
     *
     * @var    boolean
     * @since  1.0
     */
    protected $isEnabled = false;

    /**
     * The regularTimesArray
     *
     * @var    array
     * @since  1.0
     */
    protected $regularTimes = [
        "-1" => [],
        "0"  => [],
        "1"  => [],
        "2"  => [],
        "3"  => [],
        "4"  => [],
        "5"  => [],
        "6"  => [],
    ];

    /**
     * The regularDuration
     *
     * @var   array
     * @since  1.0
     */
    protected $regularDuration = [];

    /**
     * The excludedTimes
     *
     * @var   array
     * @since  1.0
     */
    protected $excludedDays = [];

    /**
     * The excludedTimes
     *
     * @var   array
     * @since  1.0
     */
    protected $excludedTimeFrames = [];

    /**
     * The excludedTimes
     *
     * @var   array
     * @since  1.0
     */
    protected $excludedTimes = [];

    /**
     * The excludedTimes By Caldav
     *
     * @var   null|array
     * @since  1.0
     */
    protected $excludedCalDAVTimes = null;

    /**
     * The includedTimes
     *
     * @var   array
     * @since  1.0
     */
    protected $includedTimeFrames = [];

    /**
     * The excludedTimes by bookings
     *
     * @var   array
     * @since  1.0
     */
    protected $appointmentBookings = [];

    /**
     * The excluded Days by max per day
     *
     * @var   array
     * @since  1.0
     */
    protected $appointmentBookingsFullDay = [];

    /**
     * The bookabel Start
     *
     * @var   \DateTime
     * @since  1.0
     */
    protected $bookableStart;

    /**
     * Is Captcha Enabled
     *
     * @var   Boolean
     * @since  1.0
     */
    protected $captchaEnabled = false;

    /**
     * Captcha plugin
     *
     * @var   Boolean
     * @since  1.0
     */
    protected $captchaPlugin = '';

    /**
     * Appointment Settings
     *
     * @var   stdClass
     * @since  1.0
     */
    protected $appointmentSettings = null;

    /**
     * Clocking for appointment selection
     *
     * @var   int
     * @since  1.0
     */
    protected $clocking = 60;

    /**
     * appointment duration
     *
     * @var   int
     * @since  1.0
     */
    protected $duration = 60;

    /**
     * appointment duration
     *
     * @var   OutputController
     * @since  1.0.5
     */
    private $_cache;

    /**
     * logging enabled
     *
     * @var   boolean
     * @since  1.0.6
     */
    protected $log_enabled = false;

    /**
     * Method to instantiate the SismoappointmentHelper class object.
     *
     * @param   array  $data        The module data
     *
     * @since  1.0.5
     */
    public function __construct($data)
    {
        $this->app = Factory::getApplication();

        // Define the logger.
        Log::addLogger(['text_file' => 'mod_sismosappointment.php'], Log::ALL, ['mod_sismosappointment.php']);

        $plgAppointment       = PluginHelper::getPlugin('system', 'sismosappointment');
        $plgAppointmentParams = new Registry($plgAppointment->params);
        if ($plgAppointment && $plgAppointmentParams->get('log_on', 0)) {
            $this->log_enabled = true;
        }

        if (empty($data) && Factory::getApplication()->input->get('option') === "com_ajax") {
            $app            = Factory::getApplication();
            $input          = $app->input;
            $this->formData = $input->getData('jform');
            $data           = $this->formData;
        }
        if (\array_key_exists('moduleId', $data) && (int) $data['moduleId'] >= 0) {
            $splittedModulMenu  = explode(':', $data['moduleId']);
            $this->moduleId     = (\is_array($splittedModulMenu) && \count($splittedModulMenu) > 0) ? (int) $splittedModulMenu[0] : -1;
            $this->activeMenuId = (\is_array($splittedModulMenu) && \count($splittedModulMenu) > 1) ? (int) $splittedModulMenu[1] : -1;
        }
        $this->contactId = (\array_key_exists('contactId', $data)) ? $data['contactId'] : $data['contactid'] ;
        $this->isEnabled = $this->_isEnabled();
        if ($this->isEnabled) {
            $params  = $this->getAppointmentSettings($this->contactId);
            if (\is_null($params)) {
                $this->isEnabled = false;
                $this->log('Error could not get Appointment settings from contact id:' . $this->contactId, Log::ERROR);
                return;
            }
            $regular = $params->regular;
            foreach ($regular as $timeframe) {
                $this->regularTimes[$timeframe['rds']][] = ['rst' => $timeframe['rst'], 'ret' => $timeframe['ret']];
            }
            foreach ($this->regularTimes as $i => &$times) {
                if (\intval($i) > 0 && \intval($i) < 6) {
                    $times = array_merge($times, $this->regularTimes["-1"]);
                }
                usort($this->regularTimes[$i], function ($x, $y) {
                    if ($x['rst'] === $y['rst']) {
                        return 0;
                    }
                    return \intval(substr($x['rst'], 0, 2)) < \intval(substr($y['rst'], 0, 2)) ? -1 : 1;
                });
            }

            $this->regularDuration = $params->duration; //->get('sapmt-duration');
            $this->clocking        = $this->regularDuration['sapmt-cts'];
            // Special inlcuded
            $included = $params->include;
            foreach ($included as $inclTimeFrame) {
                $newInlcudeDay                                               = Factory::getDate($inclTimeFrame['ids'], 'UTC');
                $this->includedTimeFrames[$newInlcudeDay->format('Y-m-d')][] = ['ist' => $inclTimeFrame['ist'], 'iet' => $inclTimeFrame['iet']];
            }
            // Excluded by settings
            $excluded       = $params->exclude;
            $newExludeDays  = [];
            $newExludeTimes = [];
            foreach ($excluded as $exclTimeFrame) {
                if (!$exclTimeFrame['edsr'] || $exclTimeFrame['edsr'] == '') {
                    $newExludeDay = Factory::getDate($exclTimeFrame['eds'], 'UTC');
                    if ($exclTimeFrame['efd']) {
                        $newExludeDays[] = $newExludeDay->format('Y-m-d');
                        continue;
                    }
                    $newExludeTimes[$newExludeDay->format('Y-m-d')][] = ['est' => $exclTimeFrame['est'], 'eet' => $exclTimeFrame['eet']];
                } else {
                    $peDate          = Factory::getDate($exclTimeFrame['edsr'], 'UTC');
                    $peDate          = $peDate->add(new \DateInterval('P1D'));
                    $newExludePeriod = new \DatePeriod(Factory::getDate($exclTimeFrame['eds'], 'UTC'), new \DateInterval('P1D'), $peDate);
                    foreach ($newExludePeriod as $date) {
                        if ($exclTimeFrame['efd']) {
                            $newExludeDays[] = $date->format('Y-m-d');
                            continue;
                        }
                        $newExludeTimes[$date->format('Y-m-d')][] = ['est' => $exclTimeFrame['est'], 'eet' => $exclTimeFrame['eet']];
                    }
                }
            }
            // CALDAV integration
            if (property_exists($params, 'integration') && \is_array($params->integration) && $params->integration['caldav_enable']) {
                include JPATH_ROOT . "/plugins/system/sismosappointment/src/vendor/autoload.php";
                $this->excludedCalDAVTimes = (\is_null($this->excludedCalDAVTimes)) ? $this->getCalDAVAppointments($params->integration) : $this->excludedCalDAVTimes;
                foreach ($this->excludedCalDAVTimes as $ind => $caldavEntries) {
                    foreach ($caldavEntries as $caldavEntry) {
                        $newExludeTimes[$ind][] = $caldavEntry;
                    }
                }
            }
            $this->excludedTimes      = $params->exclude;
            $this->excludedDays       = array_merge_recursive($this->excludedDays, $newExludeDays);
            $this->excludedTimeFrames =  array_merge_recursive($this->excludedTimeFrames, $newExludeTimes);
            $this->_cleanupAndSortTimeFrames($this->excludedTimeFrames, true);
            // Booked times
            $bookings          = $this->_getAppointmentBookings();
            $excludedByBooking = [];
            foreach ($bookings as $booking) {
                $date = Factory::getDate($booking->appointment, 'UTC');
                $date->sub(new \DateInterval('PT' . (int) $this->regularDuration['sapmt-rpdus'] . 'M'));
                $beTime                                      = Factory::getDate($booking->appointment, 'UTC');
                $interval                                    = 'PT' . (int) $booking->duration . 'M';
                $beTime                                      = $beTime->add(new \DateInterval($interval));
                $excludedByBooking[$date->format('Y-m-d')][] = ['est' => $date->format('H:i'), 'eet' => $beTime->format('H:i')];
            }
            // TODO Kontingent
            $this->_cleanupAndSortTimeFrames($excludedByBooking);
            $newExludeDaysMax = [];
            $maxPerDay        = \intval($this->regularDuration['sapmt-mpd']);
            foreach ($excludedByBooking as $index => $exclude) {
                if (\is_array($exclude) && $maxPerDay > 0 && \count($exclude) >= \intval($this->regularDuration['sapmt-mpd'])) {
                    $newExludeDaysMax[] = $index;
                    unset($excludedByBooking[$index]);
                    continue;
                }
            }
            $this->appointmentBookings = $excludedByBooking;
            $this->_cleanupAndSortTimeFrames($this->appointmentBookings, true);
            $this->appointmentBookingsFullDay = array_unique($newExludeDaysMax);
        }
    }

    /**
     * Retrieve the form
     *
     * @param   Registry         $params  The module parameters.
     * @param   SiteApplication  $app     The Site Application object.
     *
     * @return  mixed
     *
     * @since 1.0.5
     */
    public function getForm(Registry $params, SiteApplication $app)
    {
        return $this->_getForm();
    }

    /**
     * Retrieve the form
     *
     * @return  mixed
     *
     * @since 1.0.5
     */
    private function _getForm()
    {
        if (!$this->isEnabled) {
            return false;
        }

        $model = new AppointmentModel();

        $form = $model->getForm();

        // TODO auslagern in Model
        $form->setFieldAttribute('contactId', 'default', $this->contactId, null);

        $form->setFieldAttribute('appointment_duration', 'default', $this->regularDuration['sapmt-rdus'], null);

        $form->setFieldAttribute('appointment_datetime', 'contactId', $this->contactId, null);
        $form->setFieldAttribute('appointment_datetime', 'duration', $this->regularDuration['sapmt-rdus'], null);

        $settings = $this->getAppointmentSettings($this->contactId);

        if (!$settings->form['fo_phone']) {
            $form->removeField('appointment_phone');
            $form->setFieldAttribute('appointment_message', 'rows', '4', null);
        } elseif ($settings->form['fo_phonereq']) {
            $form->setFieldAttribute('appointment_phone', 'required', 'true', null);
        }

        if ($settings->form['fo_message']) {
            $form->setFieldAttribute('additionalinfo', 'label', Text::_($settings->form['fo_message']), null);
        } else {
            $form->removeField('additionalinfo');
        }

        //Captcha
        if ($settings->form['fo_captcha']) {
            $contactParams = ComponentHelper::getParams('com_contact');

            $captchaSet = $contactParams->get('captcha', $this->app->get('captcha', '0'));

            foreach (PluginHelper::getPlugin('captcha') as $plugin) {
                if ($captchaSet === $plugin->name) {
                    $form->setFieldAttribute('captcha', 'plugin', $captchaSet, null);
                    $this->captchaEnabled = true;
                    $this->captchaPlugin  = 'plg_captcha_' . str_replace(['-','_'], '', $captchaSet);
                    break;
                }
            }
        } else {
            $form->removeField('captcha');
        }

        return $form;
    }

    public function submitFormAjax()
    {
        if (!$this->formData || empty($this->formData)) {
            $this->log('Error no form data send on Ajax sumbmission from appointments.', Log::WARNING);
            echo new JsonResponse(['field' => false,"mail" => true], Text::_('MOD_APPOINTMENT_SAVE_ERR_MESSAGE'), true);
            $this->app->close();
            return false;
        }

        $result   = ['field' => false,"mail" => true];
        $language = $this->app->getLanguage();
        $language->load('mod_sismosappointment', JPATH_BASE . '/modules/mod_sismosappointment');

        if (!\is_array($this->formData) || !$this->app->getInput()->get(Session::getFormToken(), 0)) {
            $this->log('Error form data send on Ajax sumbmission from appointments or session token invalid.', Log::WARNING);
            echo new JsonResponse($result, Text::_('MOD_APPOINTMENT_SAVE_ERR_MESSAGE'), true);
            $this->app->close();
            return false;
        }

        $form = $this->_getForm();

        if (!$form) {
            $this->log('Error could not get form on Ajax sumbmission from appointments.', Log::WARNING);
            echo new JsonResponse($result, Text::_('MOD_APPOINTMENT_SAVE_ERR_MESSAGE'), true);
            $this->app->close();
            return false;
        }

        if ($this->captchaEnabled) {
            // Captcha Plugins often use Input to validate Data and search for Honey Pot etc.
            $input = Factory::getApplication()->getInput();

            foreach ($this->formData as $key => $data) {
                $input->set($key, $data);
            }
        }

        $form->bind($this->formData);

        $timezoneOffset = (int) $this->formData['tzoffset'];

        $intervalString = ((int) $timezoneOffset < 0) ? 'PT' . ((int) ($timezoneOffset) * -1) . 'M' : 'PT' . (int) ($timezoneOffset) . 'M';
        $interval       = new \DateInterval($intervalString);

        $date = Factory::getDate($this->formData['appointment_datetime'], 'UTC');
        if ((int) $timezoneOffset < 0) {
            $date->sub($interval);
        } else {
            $date->add($interval);
        }

        $isValid           = true;
        $validate_interval = new \DateInterval('PT' . (int) $this->formData['appointment_duration'] . 'M');
        if (!$this->_validateBookingTime($date, $validate_interval)) {
            $isValid = false;
            $this->app->enqueueMessage(
                Text::_('MOD_APPOINTMENT_FIELD_APPOINTMENT_NOVALIDOPTION_ERR_MESSAGE'),
                'warning'
            );

            $form->setFieldAttribute('appointment_datetime', 'contactId', (int) $this->formData['contactId'], null);
            $form->setFieldAttribute('appointment_datetime', 'tzoffset', $timezoneOffset, null);
            $form->setFieldAttribute('appointment_datetime', 'duration', (int) $this->formData['appointment_duration'], null);

            $result['field'] = $form->renderField('appointment_datetime');
        }

        $this->formData['appointment_datetime'] = $date->toSql();

        if (\array_key_exists('appointment_phone', $this->formData) && $this->formData['appointment_phone']) {
            $this->formData['appointment_message'] .= Text::sprintf('MOD_SISMOSAPPOINTMENT_BOOKING_PHONE_MESSAGE', $this->formData['appointment_phone']);
        }

        $model     = new AppointmentModel();
        $dataValid = $model->validate($form, $this->formData);

        // Check for an error.
        if ($dataValid instanceof \Exception) {
            echo new JsonResponse($result, Text::_('MOD_APPOINTMENT_SAVE_ERR_MESSAGE'), true);
            $this->log('Submitted data from appointment ajax form is not valid: ' . $dataValid->getMessage(), Log::WARNING);
            $this->app->close();
            return false;
        }

        // Check the validation results.
        if ($isValid === false || $dataValid === false) {
            $listMessage = [];
            // Get the messages from messageQueue.
            $MessageQueue = $this->app->getMessageQueue();
            foreach ($MessageQueue as $msg) {
                $listMessage[] = $msg['message'];
            }

            // Get the validation messages from the form.
            if (!\count($listMessage)) {
                foreach ($form->getErrors() as $message) {
                    $listMessage[] = Text::_($message->getMessage());
                }
            }
            echo new JsonResponse($result, implode('<br>', $listMessage), true);
            $this->app->close();
            return false;
        }

        if ($this->_save($dataValid)) {
            // Send the email
            $helper = new \SiSmOS\Plugin\System\Sismosappointment\Helper\SismosappointmentHelper();
            $sent   = false;
            $sent   = $helper->sendEmail($dataValid);
            if (!$sent) {
                $this->log('Could not send email for appointment', Log::WARNING);
            }
            $result['mail'] = $sent;
        } else {
            echo new JsonResponse($result, Text::_('MOD_APPOINTMENT_SAVE_ERR_MESSAGE'), true);
            $this->app->close();
            return false;
        }

        return $result;
    }

    public function getAppointmentFormAjax()
    {
        $tzOffset           = (string) $this->formData['tzoffset'] ?? '0';
        $contactId          = (string) $this->formData['contactId'] ?? $this->contactId;
        $moduleSettings     = (string) $this->formData['moduleId'] ?? "-1";
        $splittedModulMenu  = explode(':', $moduleSettings);
        $this->moduleId     = (\is_array($splittedModulMenu) && \count($splittedModulMenu) > 0) ? (int) $splittedModulMenu[0] : -1;
        $this->activeMenuId = (\is_array($splittedModulMenu) && \count($splittedModulMenu) > 1) ? (int) $splittedModulMenu[1] : -1;

        $language = $this->app->getLanguage();

        $language->load('mod_sismosappointment', JPATH_BASE . '/modules/mod_sismosappointment');
        $language->load('lib_sismos', JPATH_BASE . '/libraries/sismos');

        $form = $this->_getForm();

        $form->setFieldAttribute('appointment_datetime', 'contactId', $contactId, null);
        $form->setFieldAttribute('appointment_datetime', 'moduleId', $this->moduleId, null);
        $form->setFieldAttribute('appointment_datetime', 'tzoffset', $tzOffset, null);
        $form->setFieldAttribute('appointment_datetime', 'duration', $this->regularDuration['sapmt-rdus'], null);

        //$result['field'] = $form->renderField('appointment_datetime');

        $captchaEnabled = $this->captchaEnabled;

        $moduleParams = [];
        if ((int) $this->moduleId >= 0) {
            $module = ModuleHelper::getModuleById((string) $this->moduleId);
            if ($module && $module->params) {
                $moduleParams = json_decode($module->params, true);
            }
        }

        $formLayout = ModuleHelper::getLayoutPath('mod_sismosappointment', ((\array_key_exists('layout', $moduleParams) && $moduleParams['layout']) ? $moduleParams['layout'] : 'default') . '_form');
        ob_start();
        include $formLayout;
        $html = ob_get_clean();

        $doc = Factory::getApplication()->getDocument();

        $buffer = '';
        if ($captchaEnabled) {
            // Generate inline stylesheet declarations
            foreach ($doc->_style as $type => $contents) {
                if ($type !== "text/css") {
                    continue;
                }
                // Test for B.C. in case someone still store stylesheet declarations as single string
                if (\is_string($contents)) {
                    $contents = [$contents];
                }

                foreach ($contents as $content) {
                    $buffer .= $content;
                }
            }

            /** @var Joomla\CMS\WebAsset\WebAssetManager $wa */
            $wa = $doc->getWebAssetManager();

            $captchaSet = $this->captchaPlugin;

            if ($wa->assetExists('script', $captchaSet)) {
                $asset                                =  $wa->getAsset('script', $captchaSet);
                $result['scripts'][$asset->getName()] = [$asset->getUri(), $asset->getAttributes()];
            }

            if ($wa->assetExists('script', $captchaSet . '.api')) {
                $asset                                =  $wa->getAsset('script', $captchaSet . '.api');
                $result['scripts'][$asset->getName()] = [$asset->getUri(), $asset->getAttributes()];
            }

            if ($wa->assetExists('style', $captchaSet)) {
                $asset                               =  $wa->getAsset('style', $captchaSet);
                $result['styles'][$asset->getName()] = [$asset->getUri(), $asset->getAttributes()];
            }

            $styleAssets = $wa->getAssets('style');
            foreach ($styleAssets as $asset) {
                $assetOptions = $asset->getOptions() ?? [];
                if (!\array_key_exists('inline', $assetOptions) || !$assetOptions['inline']) {
                    continue;
                }
                $result['styles'][$asset->getName()]            = [$asset->getUri(), $asset->getAttributes()];
                $result['styles'][$asset->getName()]['options'] =  $asset->getOptions();
            }
        }

        $result['modalTitle'] = (\array_key_exists('modal_title', $moduleParams) && $moduleParams['modal_title']) ? Text::_($moduleParams['modal_title']) : Text::_('MOD_SISMOSAPPOINTMENT_BOOKING_DEFAULT_LABEL');

        $html .= ($buffer) ? '<style>' . $buffer . '</style>' : '';

        $result['form'] = $html;

        return $result;
    }

    /**
     * Method to save the form data.
     *
     * @param   array  $data  The form data.
     *
     * @return  boolean  True on success.
     *
     * @since 1.0.5
     *
     * @throws  Exception
     * // TODO move to Model
     */
    private function _save(&$data)
    {

        $data['activation'] = ApplicationHelper::getHash(UserHelper::genRandomPassword());
        $data['block']      = 1;
        $data['created']    = Factory::getDate()->toSql();

        $db        = Factory::getContainer()->get('DatabaseDriver');
        $query     = $db->getQuery(true);
        $query     = $db->getQuery(true)
            ->insert($db->quoteName('#__sismos_appointment_entries'))
            ->columns(
                [
                    $db->quoteName('contact_id'),
                    $db->quoteName('name'),
                    $db->quoteName('email'),
                    $db->quoteName('message'),
                    $db->quoteName('appointment'),
                    $db->quoteName('offset'),
                    $db->quoteName('duration'),
                    $db->quoteName('activation'),
                    $db->quoteName('block'),
                    $db->quoteName('created'),
                ]
            );
        $query->values(
            implode(
                ',',
                $query->bindArray(
                    [$data['contactId'], $data['appointment_name'],$data['appointment_email'],$data['appointment_message'],$data['appointment_datetime'],$data['tzoffset'],$data['appointment_duration'], $data['activation'],$data['block'],$data['created']],
                    [ParameterType::INTEGER, ParameterType::STRING, ParameterType::STRING,ParameterType::STRING,ParameterType::STRING,ParameterType::INTEGER,ParameterType::INTEGER,ParameterType::STRING,ParameterType::INTEGER,ParameterType::STRING]
                )
            )
        );

        $db->setQuery($query);
        try {
            $db->execute();
        } catch (\Exception $e) {
            $this->log(Text::sprintf('JLIB_DATABASE_ERROR_FUNCTION_FAILED', $e->getCode(), $e->getMessage()), Log::ERROR);
            return false;
        }

        return true;
    }

    /**
     * Method to get certain otherwise inaccessible properties from the helper class.
     *
     * @param   string  $name  The property name for which to get the value.
     *
     * @return  mixed  The property value or null.
     *
     * @since 1.0.5
     */
    public function __get($name)
    {
        switch ($name) {
            case 'isEnabled':
                return $this->$name;
        }
    }

    protected function _isEnabled()
    {
        $app = Factory::getApplication();

        /** @var ContactsModel $model */
        $model = $app->bootComponent('com_contact')->getMVCFactory()->createModel('Contact', 'Site', ['ignore_request' => true]);
        // Set application parameters in model
        $model->setState('params', $app->getParams());

        try {
            $contact = $model->getItem($this->contactId);
        } catch (\Exception $ex) {
            return false;
        }

        $params = $contact->params;

        // Not in published State
        if ((int) $contact->published !== 1) {
            return false;
        }

        // TODO Access public ???
        if ($params->get('access-view')) {
            return $params->get('sapmt-enabled', 0);
        }

        // User object
        $user       = $app->getIdentity();
        $authorised = Access::getAuthorisedViewLevels($user->get('id'));

        $isEnabled = ($params->get('sapmt-enabled', 0) && \in_array($contact->access, $authorised));

        return $isEnabled;
    }

    /**
     * Method to get the Appointment Settings from plugin and merge with module settings as object.
     *
     * @param   int  $contactId  The contactId.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since 1.0.5
     */

    protected function getAppointmentSettings($contactId)
    {

        if ((int) $this->contactId !== (int) $contactId) {
            $helper = new \SiSmOS\Plugin\System\Sismosappointment\Helper\SismosappointmentHelper($contactId);

            return $helper->getAppointmentSettings();
        }

        if (\is_null($this->appointmentSettings)) {
            $helper                    = new \SiSmOS\Plugin\System\Sismosappointment\Helper\SismosappointmentHelper($this->contactId);
            $this->appointmentSettings = $helper->getAppointmentSettings();
        }

        if ($this->moduleId >= 0) {
            $data = [];
            if ($this->activeMenuId > -1) {
                Factory::getApplication()->getInput()->set('Itemid', (int) $this->activeMenuId);
            }
            $module = ModuleHelper::getModuleById((string) $this->moduleId);
            if ($module && $module->params) {
                $modParams                                          = json_decode($module->params, true);
                $data['modal_duration']                             = (\array_key_exists('modal_duration', $modParams)) ? $modParams['modal_duration'] : '-1';
                $data['modal_prep']                                 = (\array_key_exists('modal_prep', $modParams)) ? $modParams['modal_prep'] : '-1';
                $data['modal_bstart']                               = (\array_key_exists('modal_bstart', $modParams)) ? $modParams['modal_bstart'] : '-1';
                $this->appointmentSettings->duration['sapmt-rdus']  = (\array_key_exists('modal_duration', $data) && (int) $data['modal_duration'] >= 0) ? $data['modal_duration'] : $this->appointmentSettings->duration['sapmt-rdus'];
                $this->appointmentSettings->duration['sapmt-rpdus'] = (\array_key_exists('modal_prep', $data) && (int) $data['modal_prep'] >= 0) ? $data['modal_prep'] : $this->appointmentSettings->duration['sapmt-rpdus'];
                $this->appointmentSettings->duration['sapmt-fbt']   = (\array_key_exists('modal_bstart', $data) && $data['modal_bstart'] != '-1') ? $data['modal_bstart'] : $this->appointmentSettings->duration['sapmt-fbt'];
            }
        }

        return $this->appointmentSettings;
    }

    /**
     * Method to get the Appointment Settings from db as object.
     *
     * @param   DatabaseDriver|\JDatabaseDriver  $db  The DatabaseDriver.
     * @param   int  $contactId  The contactId.
     *
     * @return  stdClass  The appointment settings object.
     *
     * @since 1.0.5
     */
    private function _getAppointmentBookings()
    {
        $model = new AppointmentsModel();
        $model->setState('filter.current', true);
        $model->setState('filter.contact_id', $this->contactId);
        $model->setState('filter.block', true);

        return $model->getItems();
        ;
    }

    /**
     *
     *
     *
     * @return  array
     *
     * @since 1.0.5
     */
    public function getBookableTimeFrames()
    {

        $start = $this->_getBookableStart();

        $end = new Date('now', 'UTC');

        if ($this->regularDuration['sapmt-rdr'] <= 0) {
            $end->modify('last day of this month');
        } else {
            $end->add(new \DateInterval('P' . (int) $this->regularDuration['sapmt-rdr'] . 'M'));
        }

        $datePeriode = new \DatePeriod($start, new \DateInterval('P1D'), $end);

        $interval = 'PT' . $this->regularDuration['sapmt-rdus'] . 'M';

        $period = [];

        foreach ($datePeriode as $date) {
            if (\in_array($date->format('Y-m-d'), $this->appointmentBookingsFullDay)) {
                $period[$date->format('Y-m-d')] = '';
                continue;
            } elseif ($this->isDayDisabledByTimes($date)) {
                if (\array_key_exists($date->format('Y-m-d'), $this->includedTimeFrames)) {
                    $newTimeFrames = [];
                    foreach ($this->includedTimeFrames[$date->format('Y-m-d')] as $index => $timeFrame) {
                        $pdate         = new Date($date->format('Y-m-d') . ' ' . $timeFrame['ist'], 'UTC');
                        $pedate        = new Date($date->format('Y-m-d') . ' ' . $timeFrame['iet'], 'UTC');
                        $newTimeFrames = array_merge($newTimeFrames, $this->addTimeFrame($pdate, $pedate, new \DateInterval($interval)));
                    }
                    sort($newTimeFrames);
                    if (\array_key_exists($date->format('Y-m-d'), $period)) {
                        $period[$date->format('Y-m-d')] = array_unique(array_merge($period[$date->format('Y-m-d')], $newTimeFrames));
                    } else {
                        $period[$date->format('Y-m-d')] = $newTimeFrames;
                    }
                    continue;
                }
            } else {
                $dow = $date->format('w');
                foreach ($this->regularTimes[$dow] as $i => $timeFrame) {
                    $newTimeFrames = [];
                    $pdate         = new Date($date->format('Y-m-d') . ' ' . $timeFrame['rst'], 'UTC');
                    if ($i === 0 && ($index = $this->isSpecialIncludedBeforeRegularStartTimes($pdate)) !== false) {
                        $psidateIf = new Date($date->format('Y-m-d') . ' ' . $this->includedTimeFrames[$date->format('Y-m-d')][$index]['ist'], 'UTC');
                        if ($psidateIf >= $start) {
                            $pdate = $psidateIf;
                        }
                    }
                    $pedate           = new Date($date->format('Y-m-d') . ' ' . $timeFrame['ret'], 'UTC');
                    $newpedate        = clone $pedate; //new Date($date->format('Y-m-d') . ' ' . $timeFrame['ret']);
                    $isExcludedPeriod = $this->isPeriodCompletelyDisabledByExcludedTimes($pdate, $pedate);
                    if ($isExcludedPeriod) {
                        if (\array_key_exists($date->format('Y-m-d'), $this->includedTimeFrames)) {
                            while (($index = $this->isSpecialIncludedTimes($pdate, $pedate)) !== false) {
                                $pstart = new Date($date->format('Y-m-d') . ' ' . $this->includedTimeFrames[$date->format('Y-m-d')][$index]['ist'], 'UTC');
                                $pend   = new Date($date->format('Y-m-d') . ' ' . $this->includedTimeFrames[$date->format('Y-m-d')][$index]['iet'], 'UTC');
                                // if ($pstart >= $start) {
                                $newTimeFrames = array_merge($newTimeFrames, $this->addTimeFrame($pstart, $pend, new \DateInterval($interval)));
                                // }
                                if ($pend >= $pedate) {
                                    break;
                                }
                            }
                            sort($newTimeFrames);
                            if (\array_key_exists($date->format('Y-m-d'), $period)) {
                                $period[$date->format('Y-m-d')] = array_unique(array_merge_recursive($period[$date->format('Y-m-d')], $newTimeFrames));
                            } else {
                                $period[$date->format('Y-m-d')] = $newTimeFrames;
                            }
                        }
                        continue;
                    }
                    while (($index = $this->isDisabledByExcludedTimes($pdate, $pedate)) !== false) {
                        $newpedate = new Date($date->format('Y-m-d') . ' ' . $this->excludedTimeFrames[$date->format('Y-m-d')][$index]['est']);
                        if ($this->isPeriodeCompleteInSpecialInlcudeTimes($newpedate, new Date($date->format('Y-m-d') . ' ' . $this->excludedTimeFrames[$date->format('Y-m-d')][$index]['eet']))) {
                            break;
                        }
                        while (($nindex = $this->isSpecialIncludedTimes($newpedate, new Date($date->format('Y-m-d') . ' ' . $this->excludedTimeFrames[$date->format('Y-m-d')][$index]['eet']))) !== false) {
                            $pstart = new Date($date->format('Y-m-d') . ' ' . $this->includedTimeFrames[$date->format('Y-m-d')][$nindex]['ist'], 'UTC');
                            $pend   = new Date($date->format('Y-m-d') . ' ' . $this->includedTimeFrames[$date->format('Y-m-d')][$nindex]['iet'], 'UTC');
                            // if ($pstart >= $start) {
                            $newTimeFrames = array_merge($newTimeFrames, $this->addTimeFrame($pstart, $pend, new \DateInterval($interval)));
                            // }
                            if ($pend >= $newpedate) {
                                break;
                            }
                        }
                        sort($newTimeFrames);
                        $newTimeFrames = array_merge($newTimeFrames, $this->addTimeFrame($pdate, $newpedate, new \DateInterval($interval)));
                        $pdate         = new Date($date->format('Y-m-d') . ' ' . $this->excludedTimeFrames[$date->format('Y-m-d')][$index]['eet'], 'UTC');
                    }
                    $newTimeFrames = array_merge($newTimeFrames, $this->addTimeFrame($pdate, $pedate, new \DateInterval($interval)));
                    if (\array_key_exists($date->format('Y-m-d'), $period)) {
                        $period[$date->format('Y-m-d')] = array_unique(array_merge_recursive($period[$date->format('Y-m-d')], $newTimeFrames));
                    } else {
                        $period[$date->format('Y-m-d')] = $newTimeFrames;
                    }
                }
            }
            if (\array_key_exists($date->format('Y-m-d'), $period) && \is_array($period[$date->format('Y-m-d')]) && \count($period[$date->format('Y-m-d')]) > 0) {
                asort($period[$date->format('Y-m-d')]);
            } else {
                $period[$date->format('Y-m-d')] = '';
            }
        }

        //  while (!is_array($period[array_key_first($period)]) || count($period[array_key_first($period)]) === 0) {
        // unset($period[array_key_first($period)]);
        // }

        return $period;
    }

    /**
     * Method to get the bookabel start.
     *
     * @return  \DateTime   First Bookable Start Datetime
     *
     * @since 1.0.5
     */
    private function _getBookableStart()
    {
        if ($this->bookableStart) {
            return $this->bookableStart;
        }

        $firstBookable = $this->regularDuration['sapmt-fbt'];
        $start         = Factory::getDate('now', 'UTC');//new Date('now');
        $precision     = $this->clocking; // roundToNextClocking
        $s             = $precision * 60;
        $start->setTimestamp($s * (int) ceil($start->getTimestamp() / $s));

        if ($firstBookable === 'NW') {
            $start = Factory::getDate('00:00:00', 'UTC');
            $start->modify('next monday');
        //$start = new Date($start->format('Y-m-d') . ' 00:00:00');
        } elseif ($firstBookable === '1D' || $firstBookable === '2D' || $firstBookable === '3D') {
            $start = Factory::getDate('00:00:00', 'UTC');
            $start->add(new \DateInterval('P' . $firstBookable));
        //$start = new Date($start->format('Y-m-d') . ' 00:00:00');
        } else {
            $start->add(new \DateInterval('PT' . \intval($firstBookable) . 'M'));
        }

        // $start->add(new \DateInterval('P' . (int) 25 . 'D')); // @todo remove after testing

        $this->bookableStart = $start;
        return $start;
    }

    /**
     * Method to add timeframe to list.
     *
     * @param   \DateTime  $dateStart         The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     * @param   DateInvertal $interval  The interval
     *
     * @return  array   array timeframe
     *
     * @since 1.0.5
     */
    protected function addTimeFrame($dateStart, $dateEnd, \DateInterval $interval)
    {
        $newTimeFrames = [];
        if ($this->_isPeriodCompletelyDisabledByBookingTimes($dateStart, $dateEnd)) {
            return $newTimeFrames;
        }
        if ($this->_getBookableStart() > $dateStart) {
            $dateStart = $this->_getBookableStart();
        }

        $newPeriode   = new \DatePeriod($dateStart, new \DateInterval('PT' . (string) $this->clocking . 'M'), $dateEnd);
        $periodeArray = $this->filterExludeByBookingTimes($newPeriode, $interval);

        foreach ($periodeArray as $i => $ntf) {
            $checkEnd = new Date($ntf->format('Y-m-d H:i'), 'UTC');
            $checkEnd->add($interval);
            if ($checkEnd <= $dateEnd) { //&& $checkStart >= $dateStart) {
                $newTimeFrames[] = Factory::getDate($ntf->format('Y-m-d H:i:s'), 'UTC')->format('Y-m-d H:i:s');
                continue;
            }
        }

        return $newTimeFrames;
    }

    /**
     * Method to cleanup and sort timeframe array.
     *
     * @param   array      $array       The timesframes array
     * @param   boolean    $timeRanges  build time ranges or only cleanup and sort default false
     *
     *
     * @since 1.0.5
     */
    protected function _cleanupAndSortTimeFrames(&$array, $timeRanges = false)
    {
        foreach ($array as &$arr) {
            $arr = array_values(array_intersect_key($arr, array_unique(array_map('serialize', $arr))));
            usort($arr, function ($x, $y) {
                if ($x[array_key_first($x)] === $y[array_key_first($y)]) {
                    return 0;
                }
                return \intval(substr($x[array_key_first($x)], 0, 2)) < \intval(substr($y[array_key_first($y)], 0, 2)) ? -1 : 1;
            });
        }
        if ($timeRanges) {
            foreach ($array as $key => &$arr) {
                if (\is_array($arr) && \count($array[$key]) > 1 && \count($arr) > 1) {
                    $dateStart = new Date($key . ' ' . $arr[0][array_key_first($arr[0])], 'UTC');
                    $dateEnd   = new Date($key . ' ' . $arr[0][array_key_last($arr[0])], 'UTC');
                    $this->_getFullRangesTimeFrames($array, $dateStart, $dateEnd);
                }
                if (\is_array($arr) && \count($arr) > 1) {
                    $arr = array_values(array_intersect_key($arr, array_unique(array_map('serialize', $arr))));
                }
            }
        }
    }

    /**
     * Method to cleanup and sort timeframe array.
     *
     * @param   array       $array       The timesframes array
    *  @param   \DateTime    $dateStart   The start datetime
     * @param   \DateTime    $dateEnd     The end datetime
     *
     *
     * @since 1.0.5
     */
    protected function _getFullRangesTimeFrames(&$array, $dateStart, $dateEnd)
    {
        foreach ($array[$dateStart->format('Y-m-d')] as &$arr) {
            $checkStart = new Date($dateStart->format('Y-m-d') . ' ' . $arr[array_key_first($arr)], 'UTC');
            $checkEnd   = new Date($dateStart->format('Y-m-d') . ' ' . $arr[array_key_last($arr)], 'UTC');
            if ($dateStart == $checkStart && $dateEnd == $checkEnd) {
                continue;
            }
            if ($dateStart < $checkStart && $dateEnd > $checkEnd) {
                unset($arr);
                continue;
            }
            if ($checkStart < $dateStart && $checkEnd > $dateEnd) {
                $this->_getFullRangesTimeFrames($array, $checkStart, $checkEnd);
                break;
            }
            if ($dateStart >= $checkStart && $dateStart <= $checkEnd && $dateEnd >= $checkEnd) {
                $arr[array_key_last($arr)] = $dateEnd->format(('H:i'));
                $this->_getFullRangesTimeFrames($array, $checkStart, $dateEnd);
                break;
            }
            if ($dateStart <= $checkStart && $dateEnd <= $checkEnd && $dateEnd >= $checkStart) {
                $arr[array_key_first($arr)] = $dateStart->format(('H:i'));
                $this->_getFullRangesTimeFrames($array, $dateStart, $checkEnd);
                break;
            }
        }
    }

    /**
     * Method to check if timeframe is disabled by excluded times.
     *
     * @param   \DateTime  $dateStart         The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean  check timeframe
     *
     * @since 1.0.5
     */
    protected function isPeriodCompletelyDisabledByExcludedTimes($dateStart, $dateEnd)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->excludedTimeFrames)) {
            foreach ($this->excludedTimeFrames[$key] as $index => $excludedTimes) {
                $start = new Date($key . ' ' . $excludedTimes['est'], 'UTC');
                $end   = new Date($key . ' ' . $excludedTimes['eet'], 'UTC');
                /* $test = $end >= $dateEnd;
                $test2 = $start <= $dateStart; */
                if ($start <= $dateStart && $end >= $dateEnd) {
                    return true;
                    break;
                }
            }
        }
        return false;
    }

    /**
     * Method to check if timeframe is disabled by booked times.
     *
     * @param   \DateTime  $dateStart         The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean  check timeframe
     *
     * @since 1.0.5
     */
    protected function _isPeriodCompletelyDisabledByBookingTimes($dateStart, $dateEnd)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->appointmentBookings)) {
            foreach ($this->appointmentBookings[$key] as $index => $bookedTimes) {
                $start = new Date($key . ' ' . $bookedTimes['est'], 'UTC');
                $end   = new Date($key . ' ' . $bookedTimes['eet'], 'UTC');
                if ($start <= $dateStart && $end >= $dateEnd) {
                    return true;
                    break;
                }
            }
        }
        return false;
    }

    /**
     * Method to exlude timeframes by booked times.
     *
     * @param   \DatePeriod   $periode        The Time-Frame-Periode
     * @param   DateInvertal  $interval       The interval
     *
     * @return  array  The periode as filtered array
     *
     * @since 1.0.5
     */
    protected function filterExludeByBookingTimes($periode, $interval)
    {
        $periodeArray = [];
        $added        = [];
        foreach ($periode as $ntf) {
            $periodeArray[] = $ntf;
        }
        foreach ($periodeArray as $i => &$ntf) {
            $key = $ntf->format('Y-m-d');
            if (!\array_key_exists($key, $this->appointmentBookings)) {
                continue;
            }
            $dateStart = new Date($ntf->format('Y-m-d H:i'), 'UTC');
            $dateEnd   = clone $dateStart;
            $dateEnd->add($interval);
            foreach ($this->appointmentBookings[$key] as $index => $bookedTimes) {
                $start = new Date($key . ' ' . $bookedTimes['est'], 'UTC');
                $end   = new Date($key . ' ' . $bookedTimes['eet'], 'UTC');
                $end->add(new \DateInterval('PT' . (int) $this->regularDuration['sapmt-rpdus'] . 'M'));
                if ($start >= $dateStart && $start < $dateEnd) {
                    // Start exlude beetween periode
                    unset($periodeArray[$i]);
                    $new = clone $start->sub($interval);
                    if ($new >= $periode->getStartDate() && !\in_array($new, $periodeArray) && !\in_array($new, $added)) {
                        $periodeArray[] = clone $new;
                        $added[]        = clone $new;
                    }
                } elseif ($end <= $dateEnd && $end > $dateStart) {
                    // End exlude beetween periode
                    unset($periodeArray[$i]);
                    $new = clone $end;
                    if (!\in_array($new, $periodeArray) && !\in_array($new, $added)) {
                        $periodeArray[] = clone $new;
                        $added[]        = clone $new;
                    }
                } elseif ($start <= $dateStart && $end >= $dateEnd) {
                    // complete in periode
                    unset($periodeArray[$i]);
                    $newA[] = clone $end;
                    $newA[] = clone $start->sub($interval);
                    foreach ($newA as $new) {
                        if ($new >= $periode->getStartDate() && !\in_array($new, $periodeArray) && !\in_array($new, $added)) {
                            $periodeArray[] = clone $new;
                            $added[]        = clone $new;
                        }
                    }
                }
            }
        }
        return $periodeArray;
    }

    /**
     * Validate the booking time before save.
     *
     * @param   Date      $dateStart      The start datetime
     * @param   DateInvertal  $interval       The interval
     *
     * @return  boolean   valid
     *
     * @since 1.0.5
     */
    private function _validateBookingTime($dateStart, $interval)
    {
        $key = $dateStart->format('Y-m-d');
        if (!\array_key_exists($key, $this->appointmentBookings)) {
            return true;
        }
        $dateEnd = clone $dateStart;
        $dateEnd->add($interval);
        foreach ($this->appointmentBookings[$key] as $index => $bookedTimes) {
            $start = new Date($key . ' ' . $bookedTimes['est'], 'UTC');
            $end   = new Date($key . ' ' . $bookedTimes['eet'], 'UTC');
            $end->add(new \DateInterval('PT' . (int) $this->regularDuration['sapmt-rpdus'] . 'M'));
            if (($start >= $dateStart && $start < $dateEnd) || ($end <= $dateEnd && $end > $dateStart) || ($start <= $dateStart && $end >= $dateEnd)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Method to check if timeframe is special inlcude.
     *
     * @param   \DateTime  $dateStart         The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean  check timeframe
     *
     * @since 1.0.5
     */
    protected function isPeriodeCompleteInSpecialInlcudeTimes($dateStart, $dateEnd)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->includedTimeFrames)) {
            foreach ($this->includedTimeFrames[$key] as $index => $includedTimes) {
                $start = new Date($key . ' ' . $includedTimes['ist'], 'UTC');
                $end   = new Date($key . ' ' . $includedTimes['iet'], 'UTC');
                if ($start <= $dateStart && $end >= $dateEnd) {
                    return true;
                    break;
                }
            }
        }
        return false;
    }

    /**
     * Method to check if timeframe is disabled by excluded times.
     *
     * @param   \DateTime  $dateStart         The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean|int  check timeframe returns false or index.
     *
     * @since 1.0.5
     */
    protected function isDisabledByExcludedTimes($dateStart, $dateEnd)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->excludedTimeFrames)) {
            foreach ($this->excludedTimeFrames[$key] as $index => $excludedTimes) {
                $start = new Date($key . ' ' . $excludedTimes['est'], 'UTC');
                $end   = new Date($key . ' ' . $excludedTimes['eet'], 'UTC');
                if ($start >= $dateStart && $start <= $dateEnd) {
                    return $index;
                }
                if ($end > $dateStart && $start <= $dateStart) {
                    return $index;
                }
            }
        }

        return false;
    }

    /**
     * Method to check if timeframe is included by special times.
     *
     * @param   \DateTime  $dateStart    The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean|int  check timeframe returns false or index.
     *
     * @since 1.0.5
     */
    protected function isSpecialIncludedTimes($dateStart, $dateEnd)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->includedTimeFrames)) {
            foreach ($this->includedTimeFrames[$key] as $index => $inlcudedTimes) {
                $start = new Date($key . ' ' . $inlcudedTimes['ist'], 'UTC');
                $end   = new Date($key . ' ' . $inlcudedTimes['iet'], 'UTC');
                if ($start >= $dateStart && $start <= $dateEnd) {
                    return $index;
                }
                if ($end > $dateStart && $start <= $dateStart) {
                    return $index;
                }
            }
        }

        return false;
    }

    /**
     * Method to check if timeframe is included by special times.
     *
     * @param   \DateTime  $dateStart    The start datetime
     * @param   \DateTime  $dateEnd      The end datetime
     *
     * @return  boolean|int  check timeframe returns false or index.
     *
     * @since 1.0.5
     */
    protected function isSpecialIncludedBeforeRegularStartTimes($dateStart)
    {
        $key = $dateStart->format('Y-m-d');
        if (\array_key_exists($key, $this->includedTimeFrames)) {
            foreach ($this->includedTimeFrames[$key] as $index => $inlcudedTimes) {
                $start = new Date($key . ' ' . $inlcudedTimes['ist'], 'UTC');
                if ($start < $dateStart) {
                    return $index;
                }
            }
        }

        return false;
    }

    /**
     * Method to check if full day is disabled by regular times.
     *
    * @param   \DateTime  $date         The date
     *
     * @return  boolean  check disbled day.
     *
     * @since 1.0.5
     */
    protected function isDayDisabledByTimes($date)
    {
        // Excluded by User
        if (\in_array($date->format('Y-m-d'), $this->excludedDays)) {
            return true;
        }
        $w               = $date->format('w');
        $dayHasNoEntries = (!\array_key_exists(\intval($w), $this->regularTimes) || !\is_array($this->regularTimes[\intval($w)]) || \count($this->regularTimes[\intval($w)]) === 0);
        if ($dayHasNoEntries) {
            return true;
        }

        return false;
    }

    /**
     * Ical / CalAV get Appointments
     *
     * @return  array
     */
    protected function getCalDAVAppointments($integrationSettings)
    {

        $url = (\array_key_exists('caldav_url', $integrationSettings)) ? $integrationSettings['caldav_url'] : '';

        if (!$url) {
            return [];
        }

        $cache      = (\array_key_exists('caldav_cache', $integrationSettings)) ? (int) $integrationSettings['caldav_cache'] : 3;
        $user       = (\array_key_exists('caldav_user', $integrationSettings)) ? $integrationSettings['caldav_user'] : '';
        $pw         = (\array_key_exists('caldav_password', $integrationSettings)) ? $integrationSettings['caldav_password'] : '';
        $prepBefore = (\array_key_exists('caldav-rpdus', $integrationSettings)) ? (int) $integrationSettings['caldav-rpdus'] : 0;
        $prepAfter  = (\array_key_exists('caldav-rpdusa', $integrationSettings)) ? (int) $integrationSettings['caldav-rpdusa'] : 0;

        if ($cache) {
            $chachedResult = $this->getCachedCalDAV($cache);
            if (\is_array($chachedResult)) {
                return $chachedResult;
            }
        }

        $excludedTimes  = [];
        $timespanFilter = $this->getCalDAVDateFilter();

        if ((int) $integrationSettings['caldav_enable'] == 1) {
            try {
                $http     = (new HttpFactory())->getHttp();
                $response = $http->get(str_replace('webcal://', 'https://', $url), []);
            } catch (\RuntimeException $exception) {
                $this->log(\sprintf('An error occurred while processing the external Ical Data: %s', $exception->getMessage()), Log::WARNING);
                $this->_sendErrorMessage(\sprintf('Error Connection ICal for contactId: %s', $this->contactId), \sprintf('An error occurred while processing the external Ical Data for contactId: %s Error: %s', $this->contactId, $exception->getMessage()));
                return false;
            }
        }

        if ((int) $integrationSettings['caldav_enable'] == 2) {
            $data =
            '<c:calendar-query xmlns:d="DAV:" xmlns:c="urn:ietf:params:xml:ns:caldav">
    <d:prop>
        <d:getetag />
        <c:calendar-data />
    </d:prop>
    <c:filter>
        <c:comp-filter name="VCALENDAR">
            <c:comp-filter name="VEVENT">
                <c:time-range start="' . $timespanFilter['startFilter'] . '" end="' . $timespanFilter['endFilter'] . '"/>
            </c:comp-filter>
        </c:comp-filter>
    </c:filter>
</c:calendar-query>'
            ;

            try {
                // create a Laminas stream and write the body
                $streamData = new Stream('php://memory', 'r+');
                $streamData->write($data);
                $streamData->rewind();

                // build a PSR-7 request with Laminas\Diactoros
                $request = new Request($url, 'REPORT', $streamData);

                $headers = [
                    "Accept"       => "text/xml",
                    "Content-Type" => "text/xml; charset=utf-8",
                    'Depth'        => '1',
                    'Prefer'       => 'return-minimal',
                    // cannot be lowercase @see libraries/src/Http/Transport/CurlTransport.php line 87
                    // @see https://github.com/joomla-framework/oauth2/pull/20
                ];

                $options = [
                    'headers' => $headers,
                ];

                if ($user && $pw) {
                    $options = array_merge($options, ['userauth' => $user, 'passwordauth' => $pw]);
                }

                $request = $request->withoutHeader('Host');

                $http = (new HttpFactory())->getHttp($options, 'Stream');

                $response = $http->sendRequest($request);
            } catch (\RuntimeException $exception) {
                $this->log(\sprintf('An error occurred while processing the external CalDav Data: %s', $exception->getMessage()), Log::WARNING);
                $this->_sendErrorMessage(\sprintf('Error Connection CalDav for contactId: %s', $this->contactId), \sprintf('An error occurred while processing the external CalDav Data for contactId: %s Error: %s', $this->contactId, $exception->getMessage()));
                return false;
            }
        }

        if (!($response->code >= 200 || $response->code < 400)) {
            $this->log(\sprintf(
                'Error code %s received get appointments from external calendar: %s.',
                $response->code,
                $response->body
            ), Log::WARNING);
            $this->_sendErrorMessage(\sprintf('Error on CalDAV conection for contactId: %s', $this->contactId), \sprintf('Error code %s received get appointments from external calendar. contactId: %s Error: %s', $response->code, $this->contactId, $response->body));
        }

        if ((int) $integrationSettings['caldav_enable'] == 2 && $response->code == '207') {
            try {
                $parser = xml_parser_create_ns('UTF-8');
                xml_parser_set_option($parser, XML_OPTION_SKIP_WHITE, 1);
                xml_parser_set_option($parser, XML_OPTION_CASE_FOLDING, 0);
                if (xml_parse_into_struct($parser, $response->body, $xmlnodes, $xmltags) === 0) {
                    $this->log(\sprintf(
                        'Error parsing received external calendar. ContactId: %s.',
                        $this->contactId
                    ), Log::WARNING);
                    $this->_sendErrorMessage(\sprintf('Error on CalDAV conection for contactId: %s', $this->contactId), \sprintf('Error parsing received external calendar. ContactId: %s.', $this->contactId));
                }

                $report = [];
                $item   = [];
                if ($xmlnodes) {
                    foreach ($xmlnodes as $k => $v) {
                        switch ($v['tag']) {
                            case 'DAV::response':
                                if ($v['type'] == 'open') {
                                    $item = [];
                                } elseif ($v['type'] == 'close') {
                                    $report[] = $item;
                                }
                                break;
                            case 'DAV::href':
                                $item['href'] = basename(rawurldecode($v['value']));
                                break;
                            case 'DAV::getetag':
                                $item['etag'] = preg_replace('/^"?([^"]+)"?/', '$1', $v['value']);
                                break;
                            case 'urn:ietf:params:xml:ns:caldav:calendar-data':
                                $item['data'] = $v['value'];
                                break;
                        }
                    }

                    $tz          = new \DateTimeZone('UTC');
                    $startFilter = Factory::getDate('00:00:00', 'UTC');
                    $endFilter   = clone $startFilter;
                    if ($this->regularDuration['sapmt-rdr'] <= 0) {
                        $endFilter->modify('last day of this month');
                    } else {
                        $endFilter->add(new \DateInterval('P' . (int) $this->regularDuration['sapmt-rdr'] . 'M'));
                    }
                    foreach ($report as $entry) {
                        $vcalendar = Reader::read($entry['data']);
                        $vcalendar = $vcalendar->expand($startFilter, $endFilter);
                        foreach ($vcalendar->VEVENT as $ev) {
                            $startOrigin = $ev->DTSTART->getDateTime();
                            $endOrigin   = $ev->DTEND->getDateTime();

                            $start = Factory::getDate($startOrigin->format('Ymd H:i:s'));
                            $end   = Factory::getDate($endOrigin->format('Ymd H:i:s'));

                            if ($prepBefore) {
                                $start->sub(new \DateInterval('PT' . $prepBefore . 'M'));
                            }
                            if ($prepAfter) {
                                $end->add(new \DateInterval('PT' . $prepAfter . 'M'));
                            }

                            $start = $start->setTimezone($tz);
                            $end   = $end->setTimezone($tz);

                            $newExcludeTimes = $this->excludeDateTimesFromExternal($start, $end);
                            $excludedTimes   = array_merge_recursive($excludedTimes, $newExcludeTimes);
                        }
                    }
                }
            } catch (\Exception $e) {
                $this->log(\sprintf('Error reading CalDav response: %s', $e->getMessage()), Log::WARNING);
                $this->_sendErrorMessage(\sprintf('Error reading CalDav response for contactId: %s', $this->contactId), \sprintf('Error reading CalDav response for contactId: %s. Error: %s', $this->contactId, $e->getMessage()));
            }
        }

        if ((int) $integrationSettings['caldav_enable'] == 1 && $response->code == 200) {
            try {
                $tz          = new \DateTimeZone('UTC');
                $vcalendar   = Reader::read($response->body);
                $startFilter = Factory::getDate('00:00:00', 'UTC');
                $endFilter   = clone $startFilter;
                if ($this->regularDuration['sapmt-rdr'] <= 0) {
                    $endFilter->modify('last day of this month');
                } else {
                    $endFilter->add(new \DateInterval('P' . (int) $this->regularDuration['sapmt-rdr'] . 'M'));
                }
                $vcalendar = $vcalendar->expand($startFilter, $endFilter);
                foreach ($vcalendar->VEVENT as $ev) {
                    $startOrigin = $ev->DTSTART->getDateTime();
                    $endOrigin   = $ev->DTEND->getDateTime();

                    $start = Factory::getDate($startOrigin->format('Ymd H:i:s'));
                    $end   = Factory::getDate($endOrigin->format('Ymd H:i:s'));

                    if ($prepBefore) {
                        $start->sub(new \DateInterval('PT' . $prepBefore . 'M'));
                    }
                    if ($prepAfter) {
                        $end->add(new \DateInterval('PT' . $prepAfter . 'M'));
                    }

                    $start = $start->setTimezone($tz);
                    $end   = $end->setTimezone($tz);

                    $newExcludeTimes = $this->excludeDateTimesFromExternal($start, $end);
                    $excludedTimes   = array_merge_recursive($excludedTimes, $newExcludeTimes);
                }
            } catch (\Exception $e) {
                $this->log(\sprintf('Error reading ICal response: %s', $e->getMessage()), Log::WARNING);
                $this->_sendErrorMessage(\sprintf('Error reading ICal response for contactId: %s', $this->contactId), \sprintf('Error reading ICal response for contactId: %s. Error: %s', $this->contactId, $e->getMessage()));
            }
        }

        if ($cache) {
            $dt     = Factory::getDate();
            $minute = (int) $dt->format("i") % $cache;
            $dt->add(new \DateInterval("PT" . ($cache - $minute) . "M"));
            $cacheId = 'calDav.' . '.' . $timespanFilter['startFilter'] . '.' . $timespanFilter['endFilter'] . '.' . $this->contactId . '.' . $dt->format('Y-m-d H:i');
            $this->_cache->setCaching(true);
            $this->_cache->clean();
            if (\is_array($excludedTimes) && !empty($excludedTimes)) {
                $this->_cache->store($excludedTimes, md5($cacheId), 'mod_sismosappointment');
            } else {
                $this->_cache->store([], md5($cacheId), 'mod_sismosappointment');
            }
        }

        return $excludedTimes;
    }

    /**
     * Exlude DateTimes and ranges from external calendar via caldav/ical
     * @param   \DateTime  $start    The start datetime
     * @param   \DateTime  $end      The end datetime
     *
     * @return array $excludeTimes
     */
    private function excludeDateTimesFromExternal(\DateTime $start, \DateTime $end)
    {
        $exludedTimes = [];
        // If event is on a single day
        if ($start->format('Y-m-d') === $end->format('Y-m-d')) {
            $excludedTimes[$start->format('Y-m-d')][] = [
                'est' => $start->format('H:i'),
                'eet' => $end->format('H:i'),
            ];
        } else {
            // Multi-day event
            $current = clone $start;
            $lastDay = $end->format('Y-m-d');
            while ($current->format('Y-m-d') <= $lastDay) {
                if ($current->format('Y-m-d') === $start->format('Y-m-d')) {
                    // First day: from start time to end of day
                    $excludedTimes[$current->format('Y-m-d')][] = [
                        'est' => $start->format('H:i'),
                        'eet' => '23:59',
                    ];
                } elseif ($current->format('Y-m-d') === $end->format('Y-m-d')) {
                    // Last day: from midnight to end time
                    $excludedTimes[$current->format('Y-m-d')][] = [
                        'est' => '00:00',
                        'eet' => $end->format('H:i'),
                    ];
                } else {
                    // Intermediate day: full day
                    $excludedTimes[$current->format('Y-m-d')][] = [
                        'est' => '00:00',
                        'eet' => '23:59',
                    ];
                }
                $current->add(new \DateInterval('P1D'));
            }
        }

        return $excludedTimes;
    }

    private function getCachedCalDAV($expire)
    {
        try {
            /** @var OutputController $cache */
            $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                ->createCacheController('output', ['defaultgroup' => 'mod_sismosappointment']);
            $dt     = Factory::getDate();
            $minute = (int) $dt->format("i") % $expire;
            $dt->add(new \DateInterval("PT" . ($expire - $minute) . "M"));
            $timespanFilter = $this->getCalDAVDateFilter();
            $cacheId        = 'calDav.' . '.' . $timespanFilter['startFilter'] . '.' . $timespanFilter['endFilter'] . '.' . $this->contactId . '.' . $dt->format('Y-m-d H:i');

            $this->_cache->setCaching(true);
            $calDAVTimes = $this->_cache->get(md5($cacheId), 'mod_sismosappointment');
            return $calDAVTimes;
        } catch (\RuntimeException $e) {
            $this->log(\sprintf('Cache loading error for ical / caldav support: %s', $e->getMessage()), Log::WARNING);

            return false;
        }
    }

    private function getCalDAVDateFilter()
    {
        $startDate   = Factory::getDate('00:00:00', 'UTC');
        $startFilter = $startDate->format('Ymd') . 'T' . $startDate->format('His') . 'Z';

        $endDateTime = clone $startDate;
        if ($this->regularDuration['sapmt-rdr'] <= 0) {
            $endDateTime->modify('last day of this month');
        } else {
            $endDateTime->add(new \DateInterval('P' . (int) $this->regularDuration['sapmt-rdr'] . 'M'));
        }

        $endFilter = $endDateTime->format('Ymd') . 'T' . $endDateTime->format('His') . 'Z';

        return ['startFilter' => $startFilter, 'endFilter' => $endFilter];
    }

    /**
     * Method to send an error message to Admin.
     *
     * @param   string    $subject   The subject for the message.
     * @param   string    $message   The message text
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.0.7
     */
    private function _sendErrorMessage($subject, $message)
    {
        if (\JDEBUG) {
            $this->app->enqueueMessage($message, 'error');
            $this->log($message, Log::DEBUG);
        }

        // Messaging to admin on Error
        // Push a notification to the site's super users, send email to user fails so the below message goes out
        /** @var MessageModel $messageModel */
        $messageModel = $this->app->bootComponent('com_messages')->getMVCFactory()->createModel('Message', 'Administrator');

        $messageModel->notifySuperUsers(
            $subject,
            $message
        );
    }

    /**
     * Log helper function
     *
     * @return  string
     */
    public function log($msg, $type)
    {
        if ($this->log_enabled) {
            Log::add($msg, $type, 'mod_sismosappointment');
        }
    }
}
