<?php

/**
 * @package     SISMOSAppointments
 * @subpackage  Sismos.Zoom
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\Sismosappointment\Zoom\Field;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\SubformField;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * The Field to show, create and refresh a oauth token
 *
 * @since  1.0.0
 */
class TokenField extends SubformField
{
    /**
     * The form field type.
     * @var    string
     */
    protected $type = 'Token';

    /**
     * Method to attach a Form object to the field.
     *
     * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the <field /> tag for the form field object.
     * @param   mixed              $value    The form field value to validate.
     * @param   string             $group    The field name group control value.
     *
     * @return  boolean  True on success.
     *
     * @since   1.0.0
     */
    public function setup(\SimpleXMLElement $element, $value, $group = null)
    {
        /**
         * When you have subforms which are not repeatable (i.e. a subform custom field with the
         * repeat attribute set to 0) you get an array here since the data comes from decoding the
         * JSON into an associative array, including the media subfield's data.
         *
         * However, this method expects an object or a string, not an array. Typecasting the array
         * to an object solves the data format discrepancy.
         */
        $value = \is_array($value) ? (object) $value : $value;

        /**
         * If the value is not a string, it is
         * most likely within a custom field of type subform
         * and the value is a stdClass with properties
         * access_token. So it is fine.
        */
        if (\is_string($value)) {
            json_decode($value);

            // Check if value is a valid JSON string.
            if ($value !== '' && json_last_error() !== JSON_ERROR_NONE) {
                $value = '';
            }
        } elseif (
            !\is_object($value)
            || !property_exists($value, 'access_token')
        ) {
            $value->access_token = "ERROR";
        }

        if (!parent::setup($element, $value, $group)) {
            $value = '';
        }

        $xml = <<<XML
<?xml version="1.0" encoding="utf-8"?>
<form>
	<fieldset
		name="token"
		label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_LABEL"
	>
		<field
			name="access_token"
			type="textarea"
			cols="50"
			rows="2"
			default=""
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_LABEL"
			description="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_DESC"
			readonly="true"
			filter="raw"
		/>

		<field
			name="token_type"
			type="text"
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_TYPE_LABEL"
			readonly="true"
		/>

		<field
			name="expires_in"
			type="text"
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_EXPIRES_LABEL"
			readonly="true"
		/>

		<field
			name="scope"
			type="text"
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_SCOPE_LABEL"
			readonly="true"
		/>

		<field
			name="created"
			type="text"
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_CREATED_LABEL"
			readonly="true"
		/>

		<field
			name="refresh_token"
			type="textarea"
			cols="50"
			rows="2"
			label="PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_REFRESH_LABEL"
			readonly="true"
			filter="raw"
		/>
	</fieldset>
</form>
XML;
        $this->formsource = $xml;

        return true;
    }

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     *
     * @since   1.0.0
     */
    protected function getInput()
    {
        /** @var Joomla\CMS\WebAsset\WebAssetManager $wa */
        $wa = Factory::getApplication()->getDocument()->getWebAssetManager();
        $wa->registerAndUseStyle('plg_sismosappointment_zoom.token', 'plg_sismosappointment_zoom/sismos_token.css')
            ->registerAndUseScript('plg_sismosappointment_zoom.token', 'plg_sismosappointment_zoom/sismos_token.js');

        $input = parent::getInput();
        $input .= '<div class="d-flex float-end">';
        $input .= '<button type="button" id ="genToken" class="btn btn-success me-3">' . Text::_('PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_ACTION') . '</button>';
        $input .= '<button type="button" id ="clearToken" class="btn btn-danger">' . Text::_('PLG_SISMOSAPPOINTMENT_ZOOM_TOKEN_CLEAR_ACTION') . '</button>';
        $input .= '</div>';
        $input .= '<input type="hidden" value="" name="gentoken" />';
        Text::script('PLG_SISMOSAPPOINTMENT_ZOOM_AUTH_MISSING_DATA_ERROR');
        return $input;
    }
}
