<?php

/**
 * @package     SISMOSAppointments
 * @subpackage  Sismos.Nctalk
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\Sismosappointment\NCTalk\Extension;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseDriver;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * SimplySmart-IT Appointment Nextcloud-Talk Integration.
 *
 * @package     SISMOSAppointments
 * @subpackage  Sismos.Nctalk
 * @since       1.0.1
 */
class Nctalk extends CMSPlugin implements SubscriberInterface
{
    /**
     * Application object
     *
     * @var    CMSApplication
     * @since  1.0.0
     */
    protected $app;

    /**
     * Database object
     *
     * The database is injected by parent constructor
     *
     * @var    DatabaseDriver|\JDatabaseDriver
     * @since  1.0.0
     */
    protected $db;

    /**
     * Affects constructor behavior. If true, language files will be loaded automatically.
     *
     * @var    boolean
     * @since  1.0
     */
    protected $autoloadLanguage = true;

    /**
     * Nextcloud User.
     *
     * @var    string
     * @since  1.0
     */
    private $user = '';

    /**
     * Nextcloud App-Password.
     *
     * @var    string
     * @since  1.0
     */
    private $apppw = '';

    /**
    * The http factory
    *
    * @var    HttpFactory
    * @since  1.0.0
    */
    private $httpFactory;

    /**
     * Constructor
     *
     * @param   DispatcherInterface  &$subject  The object to observe
     * @param   array                $config    An optional associative array of configuration settings.
     *                                          Recognized key values include 'name', 'group', 'params', 'language'
     *                                         (this list is not meant to be comprehensive).
     *
     * @since   1.0.0
     */
    public function __construct(&$subject, $config = [])
    {
        parent::__construct($subject, $config);

        // Define the logger.
        Log::addLogger(['text_file' => 'plg_sismosappointment_nctalk.php'], Log::ALL, ['plg_sismosappointment_nctalk']);
    }

    /**
     * @inheritDoc
     *
     * @return string[]
     *
     * @since 4.1.0
     *
     * @throws Exception
     */
    public static function getSubscribedEvents(): array
    {
        $app = Factory::getApplication();

        $mapping  = [];

        // Only allowed in the backend
        if ($app->isClient('administrator')) {
            // $mapping['onSismosGenToken'] = 'generateToken';
        } else {
            // $mapping['onCallbackIntegration'] = 'callbackIntegration';
            $mapping['onSismosMeetingActivate'] = 'createMeeting';
        }

        return $mapping;
    }

    /**
     * Get nctalk url for meeting
     *
     * @param   Event  $event  The event object
     *
     * @return  void
     *
     * @since   1.0.1
     *
     */
    public function createMeeting(Event $event)
    {
        $entry   = $event->getArgument('entry');
        $baseurl = $this->params->get('nextcloudbase', '');
        if (!$baseurl) {
            $this->log('Error create Nextcloud Talk Meeting - baseurl missing.', Log::ERROR);
            $event->addArgument('meeting_url', false);
            return;
        }
        $user  = (\array_key_exists('igv_user', $entry) && $entry['igv_user']) ? $entry['igv_user'] : '';
        $user  = (!$user && \array_key_exists('igv_default', $entry) && $entry['igv_default']) ? $this->params->get('user', '') : $user;
        $apppw = (\array_key_exists('igv_password', $entry) && $entry['igv_password']) ? $entry['igv_password'] : '';
        $apppw = (!$apppw && \array_key_exists('igv_default', $entry) && $entry['igv_default']) ? $this->params->get('password', '') : $apppw;

        if (!$user || !$apppw) {
            $this->log('Error create Nextcloud Talk Meeting - missing authorization data - user and password.', Log::ERROR);
            $event->addArgument('meeting_url', false);
            return;
        }

        $url = $baseurl . '/ocs/v2.php/apps/spreed/api/v4/room';

        $data = [
            "roomType" => 3,
            "roomName" => $entry['vc_title'] . ' - ' . $entry['name'] . ' - ' . Factory::getApplication()->get('sitename'),
        ];

        $data = json_encode($data);

        $options = [
            'userauth'     => $user,
            'passwordauth' => $apppw,
        ];

        $headers = [
            "accept"         => "application/json",
            "Content-Type"   => "application/json",
            "OCS-APIRequest" => "true",
            // cannot be lowercase @see libraries/src/Http/Transport/CurlTransport.php line 87
            // @see https://github.com/joomla-framework/oauth2/pull/20
        ];


        try {
            $response = HttpFactory::getHttp($options)->post($url, $data, $headers);
        } catch (Exception $e) {
            $this->log(sprintf('Error creating Nextcloud Talk for user_id %s: Error: %s', $entry['contact_id'], $e->getMessage()), Log::ERROR);
            $this->_sendErrorMessage('Error creating Nextcloud Talk Url', sprintf('Error creating Nextcloud Talk for user_id %s: Error: %s', $entry['contact_id'], $e->getMessage()));
            $event->addArgument('meeting_url', false);
            return;
        }

        if (!($response->code >= 200 || $response->code < 400)) {
            $this->log(sprintf(
                'Error code %s received create meeting: %s.',
                $response->code,
                $response->body
            ), Log::ERROR);
            $this->_sendErrorMessage('Error creating Nextcloud Talk Meeting Url', sprintf('Error creating Nextcloud Talk Meeting for user_id %s: Error code: %s Response:', $entry['contact_id'], $response->code, $response->body));
            $event->addArgument('meeting_url', false);
            return;
        }

        if ($this->isJsonResponse($response)) {
            $body = json_decode($response->body, true);
        } else {
            $this->log(sprintf(
                'Error code %s received create meeting json-error: %s.',
                $response->code,
                $response->body
            ), Log::ERROR);
            $this->_sendErrorMessage('Error creating Nextcloud Talk Meeting Url', sprintf('Error creating Nextcloud Talk Meeting for user_id %s: Error code: %s Response:', $entry['contact_id'], $response->code, $response->body));
            $event->addArgument('meeting_url', false);
            return;
        }

        if (\is_array($body) && isset($body['ocs']['data']['token'])) {
            $token       = $body['ocs']['data']['token'];
            $meeting     = $baseurl . '/index.php/call/' . $token;
            $this->user  = $user;
            $this->apppw = $apppw;
            $this->activateLobby($token);
        } else {
            $meeting = false;
        }

        $event->addArgument('meeting_url', $meeting);
    }

    /**
     * Activate Lobby for Talk meeting
     *
     * @param   string  $token  The meeting token
     *
     * @return  void
     *
     * @since   1.0.0
     */
    private function activateLobby($token)
    {
        $url = $this->params->get('nextcloudbase') . '/ocs/v2.php/apps/spreed/api/v4/room/' . $token . '/webinar/lobby';

        // TODO timespamp -> timer @see https://nextcloud-talk.readthedocs.io/en/latest/webinar/
        $data = [
            "state" => 1,
        ];

        $data = json_encode($data);

        $options = [
            'userauth'     => $this->user,
            'passwordauth' => $this->apppw,
        ];
        $headers = [
            "accept"         => "application/json",
            "Content-Type"   => "application/json",
            "OCS-APIRequest" => "true",
            // cannot be lowercase @see libraries/src/Http/Transport/CurlTransport.php line 87
            // @see https://github.com/joomla-framework/oauth2/pull/20
        ];

        try {
            $response = HttpFactory::getHttp($options)->put($url, $data, $headers);
        } catch (Exception $e) {
            $this->log(sprintf('Error activate lobby for Nextcloud Talk Meeting: Error: %s', $e->getMessage()), Log::ERROR);
            return;
        }

        if (!($response->code >= 200 || $response->code < 400)) {
            $this->log(sprintf(
                'Error code %s received activate lobby for meeting: %s.',
                $response->code,
                $response->body
            ), Log::ERROR);
        }
    }

    /**
     * Tests if given response contains JSON header
     *
     * @param   \Joomla\Http\Response  $response  The response object
     *
     * @return  bool
     *
     * @since   1.0.0
     * @see https://github.com/joomla-framework/oauth2/pull/20
     */
    private function isJsonResponse(\Joomla\Http\Response $response): bool
    {
        foreach ($response->getHeader('Content-Type') as $value) {
            if (strpos($value, 'application/json') !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Method to send an error message to Admin.
     *
     * @param   string    $subject   The subject for the message.
     * @param   string    $message   The message text
     *
     * @return  boolean  True on success sending the email, false on failure.
     *
     * @since   1.6.4
     */
    // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
    private function _sendErrorMessage($subject, $message)
    {

        // Messaging to admin on Error

        // Push a notification to the site's super users, send email to user fails so the below message goes out
        /** @var MessageModel $messageModel */
        $messageModel = $this->app->bootComponent('com_messages')->getMVCFactory()->createModel('Message', 'Administrator');

        $messageModel->notifySuperUsers(
            $subject,
            $message
        );
    }

    /**
     * Log helper function
     *
     * @return  string
     */
    public function log($msg, $type)
    {
        if ($this->params->get('log_on', 1)) {
            Log::add($msg, $type, 'plg_sismosappointment_nctalk');
        }
    }
}
