<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.sismosappointment
 *
 * @author      Martina Scholz <martina@simplysmart-it.de>
 *
 * @copyright   Copyright (C) 2023 - 2024 Martina Scholz - SimplySmart-IT <https://simplysmart-it.de>. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE
 * @link        https://simplysmart-it.de
 */

namespace SiSmOS\Plugin\System\Sismosappointment\Field;

use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * The Field to load the form inside current form
 *
 * @Example with all attributes:
 *  <field name="field-name" type="subform"
 *      formsource="path/to/form.xml" min="1" max="3" multiple="true" buttons="add,remove,move"
 *      layout="joomla.form.field.subform.repeatable-table" groupByFieldset="false" component="com_example" client="site"
 *      label="Field Label" description="Field Description" />
 *
 * @since  1.0.0
 */
class AppointmententriesField extends FormField
{
    /**
     * The form field type.
     * @var    string
     */
    protected $type = 'Appointmententries';

    /**
     * Name of the layout being used to render the field
     *
     * @var    string
     * @since  1.0.0
     */
    protected $layout = 'joomla.form.field.appointmententries.default';

    /**
     * contactId
     *
     * @var    string
     * @since  1.0.0
     */
    protected $contactId = '';

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     *
     * @since   1.0.0
     */
    protected function getInput()
    {
        // Value is in json format, so decode double-quotes for html value="..."
        if (empty($this->value)) {
            $this->value = '[]';
        }

        $this->value = json_decode($this->value, true);

        //Convert a date to local user time value.
        $user = Factory::getApplication()->getIdentity();

        $tz           = $user->getTimezone();
        $tzutc        = Factory::getDate('now', 'UTC');
        $tzoffsetuser = gmdate('H:i', floor(timezone_offset_get($tz, $tzutc)));

        if (\is_array($this->value)) {
            usort($this->value, function ($a, $b) {
                return (Factory::getDate($a['appointment'], 'UTC') <=> Factory::getDate($b['appointment'], 'UTC'));
            });
        }

        foreach ($this->value as &$entry) {
            // Convert a date to UTC based on the user timezone.
            if (\array_key_exists('appointment', $entry) && $entry['appointment'] != '') {
                // Get a date object based on the correct timezone.
                $utcDateTime = Factory::getDate($entry['appointment'], 'UTC');

                $utcDateTime->setTimezone($tz);

                // Transform the date string.
                $entry['appointment'] = $utcDateTime->format(Text::_('DATE_FORMAT_LC5'), true) . ' <small>UTC ' . (string) $tzoffsetuser . '</small>';
            }
            if (\array_key_exists('offset', $entry) && (int) $entry['offset'] !== 0) {
                $timezoneOffset  = $entry['offset'];
                $utcInfo         = ((int) $timezoneOffset > 0) ? '-' : '';
                $utcInfoTZOffset = ((float) $timezoneOffset < 0) ? ((float) $timezoneOffset * - 1) : (float) $timezoneOffset;
                $utcInfo .= gmdate('H:i', floor((float) ($utcInfoTZOffset * 60)));
                $entry['offset'] = 'UTC ' . $utcInfo . '</small>';
            } else {
                $entry['offset'] = 'UTC 00:00';
            }

            if (\array_key_exists('duration', $entry) && (int) $entry['duration'] !== 0) {
                $entry['duration'] = (string) gmdate('H:i', floor((float) ($entry['duration'] * 60)));
            }

            $created = Factory::getDate($entry['created'], 'UTC');
            $created->setTimezone($tz);
            $entry['created'] = HtmlHelper::date($created->format('Y-m-d H:i:s'), 'DATE_FORMAT_LC6');
        }

        //$this->value = htmlspecialchars($this->value);

        return parent::getInput();
    }

    /**
     * Method to get the field label markup.
     *
     * @return  string  The field label markup.
     *
     * @since   1.0.0
     */
    protected function getLabel()
    {
        return '';
    }

    /**
     * Method to get the data to be passed to the layout for rendering.
     *
     * @return  array
     *
     * @since 1.0.0
     */
    protected function getLayoutData()
    {
        $layoutData = parent::getLayoutData();

        $entryEditForm = new Form('plg_sismosappointment.edit');

        FormHelper::addFormPath(\dirname(__DIR__, 2) . '/forms');
        $entryEditForm->loadFile('sismosappointmententry');

        $entryEditForm->setDatabase(Factory::getDbo());

        $layoutData['entryEditForm'] = $entryEditForm;

        return $layoutData;
    }


    /**
     * Allow to override renderer include paths in child fields
     *
     * @return  array
     *
     * @since   1.0.0
     */
    protected function getLayoutPaths()
    {
        $renderer = new FileLayout($this->layout);

        $renderer->setIncludePaths(parent::getLayoutPaths());

        $renderer->addIncludePaths(\dirname(__DIR__, 2) . '/layouts');

        $paths = $renderer->getIncludePaths();

        return $paths;
    }
}
