<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Fields.datetime-local
 *
 * @copyright   (C) 2023 Martina Scholz, SimplySmart-IT <https://simplysmart-it.de>
 * @license     GNU General Public License version 3 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Sismosappointment\Field;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\CalendarField;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Form Field class for the Joomla Platform.
 *
 * Provides a pop up date picker linked to a button.
 * Optionally may be filtered to use user's or server's time zone.
 *
 * @since  1.7.0
 */
class DatetimeField extends CalendarField
{
	/**
	 * The form field type.
	 *
	 * @var    string
	 * @since  1.7.0
	 */
	protected $type = 'Datetime';
	

	/**
	 * Name of the layout being used to render the field
	 *
	 * @var    string
	 * @since  3.7.0
	 */
	protected $layout = '';

	/**
	 * The parent class of the field
	 *
	 * @var  string
	 * @since 4.0.0
	 */
	protected $parentclass;

	
	/**
	 * Method to get the field input markup.
	 *
	 * @return  string  The field input markup.
	 *
	 * @since   1.7.0
	 */
	protected function getInput()
	{
		$user = Factory::getApplication()->getIdentity();

		// If a known filter is given use it.
		switch (strtoupper($this->filter)) {
			case 'SERVER_UTC':
				// Convert a date to UTC based on the server timezone.
				if ($this->value && $this->value != $this->getDatabase()->getNullDate()) {
					// Get a date object based on the correct timezone.
					$date = Factory::getDate($this->value, 'UTC');
					$date->setTimezone(new \DateTimeZone(Factory::getApplication()->get('offset')));

					// Transform the date string.
					$this->value = $date->format('Y-m-d H:i:s', true, false);
				}
				break;
			case 'USER_UTC':
				// Convert a date to UTC based on the user timezone.
				if ($this->value && $this->value != $this->getDatabase()->getNullDate()) {
					// Get a date object based on the correct timezone.
					$date = Factory::getDate($this->value, 'UTC');
					$date->setTimezone($user->getTimezone());

					// Transform the date string.
					$this->value = $date->format('Y-m-d H:i:s', true, false);
				}
				break;
		}

		// Format value when not nulldate ('0000-00-00 00:00:00'), otherwise blank it as it would result in 1970-01-01.
		if ($this->value && $this->value != $this->getDatabase()->getNullDate() && strtotime($this->value) !== false) {
			$tz = date_default_timezone_get();
			date_default_timezone_set('UTC');

			if ($this->filterFormat) {
				$date = \DateTimeImmutable::createFromFormat('U', strtotime($this->value));
				$this->value = $date->format($this->filterFormat);
			} else {
				$this->value = strftime($this->format, strtotime($this->value));
			}

			date_default_timezone_set($tz);
		} else {
			$this->value = '';
		}

		// Handle the special case for "now".
		if (strtoupper($this->value) === 'NOW') {
			$this->value = Factory::getDate()->format('Y-m-d H:i:s');
		}

		$attributes = [];
		empty($this->size)      ? null : $attributes['size'] = $this->size;
		empty($this->maxlength) ? null : $attributes['maxlength'] = $this->maxLength;
		empty($this->class)     ? $attributes['class'] = 'form-control' : $attributes['class'] = 'form-control ' . $this->class;
		!$this->readonly        ? null : $attributes['readonly'] = 'readonly';
		!$this->disabled        ? null : $attributes['disabled'] = 'disabled';
		$description = !empty($this->description) ? ' aria-describedby="' . ($this->id ?: $this->name) . '-desc"' : '';
		
		if ($this->required) {
			$attributes['required'] = '';
		}

		if (is_array($attributes)) {
			$attributes = ArrayHelper::toString($attributes);
		}
		
		return '<input
				type="datetime-local"
				name="' . $this->name . '"
				id="' . $this->id . '"
				value="' . htmlspecialchars(($this->value !== '0000-00-00 00:00:00') ? $this->value : '', ENT_COMPAT, 'UTF-8') . '"
				' . $description . '
				' . $attributes . '>';
	}
}
