<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  mod_sismosappointment
 *
 * @copyright   Copyright (C) 2023 Martina Scholz. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt

 * @phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace
 */

 namespace Joomla\Module\Sismosappointment\Site\Field;

use DateTime;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\ListField;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Module\Sismosappointment\Site\Helper\SismosappointmentHelper;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Form Field to load a list of free Times values
 * Provides radio button inputs and slider for dates
 *
 * @since  1.0.5
 */
class AppointmentField extends ListField
{

	/**
	 * The form field type.
	 *
	 * @var    string
	 * @since  1.0.5
	 */
	protected $type = 'appointment';

	/**
	 * Name of the layout being used to render the field
	 *
	 * @var    string
	 * @since 1.0.5
	 */
	protected $layout = 'joomla.form.field.appointment';

	/**
	 * contactId
	 *
	 * @var    string
	 * @since 1.0.5
	 */
	protected $contactId = '';

	/**
	 * moduleId
	 *
	 * @var    string
	 * @since 1.0.5
	 */
	protected $moduleId = '-1';

	/**
	 * timezoneOffset
	 *
	 * @var    string
	 * @since 1.0.5
	 */
	protected $timezoneOffset = '';

	/**
	 * duration
	 *
	 * @var    string
	 * @since 1.0.5
	 */
	protected $duration = '60';

	/**
	 * Method to attach a Form object to the field.
	 *
	 * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the `<field>` tag for the form field object.
	 * @param   mixed              $value    The form field value to validate.
	 * @param   string             $group    The field name group control value. This acts as an array container for the field.
	 *                                      For example if the field has name="foo" and the group value is set to "bar" then the
	 *                                      full field name would end up being "bar[foo]".
	 *
	 * @return  boolean  True on success.
	 *
	 * @see     FormField::setup()
	 * @since 1.0.5
	 */
	public function setup(\SimpleXMLElement $element, $value, $group = null)
	{
		$return = parent::setup($element, $value, $group);

		if ($return) {
			$this->contactId       = (string) $this->element['contactId'] ? (string) $this->element['contactId'] : '';
			$this->moduleId       = (string) $this->element['moduleId'] ? (string) $this->element['moduleId'] : '-1';
			$this->timezoneOffset  = (string) $this->element['tzoffset'] ? (string) $this->element['tzoffset'] : '0';
			$this->duration        = (string) $this->element['duration'] ? (string) $this->element['duration'] : '60';
		}

		return $return;
	}

	/**
	 * Allow to override renderer include paths in child fields
	 *
	 * @return  array
	 *
	 * @since 1.0.5
	 */
	protected function getLayoutPaths()
	{
		$renderer = new FileLayout($this->layout);

		$renderer->setIncludePaths(parent::getLayoutPaths());

		$test = dirname(__DIR__, 2) . '/layouts';

		$renderer->addIncludePaths(dirname(__DIR__, 2) . '/layouts');

		$paths = $renderer->getIncludePaths();

		foreach ($paths as $index => $path) {
			if (!preg_match('#\/templates\/[a-z,A-Z,0-9,\-\_\/]*html\/layouts$#', $path, $m)) {
				continue;
			}
			$overrideTemplate = array_splice($paths, $index, 1);
			array_splice($paths, 0, 0, $overrideTemplate);
			break;
		}

		return $paths;
	}

	/**
	 * Method to get the field options.
	 *
	 * @return  array  The field option objects.
	 *
	 * @since 1.0.5
	 */
	protected function getOptions()
	{
		$fieldname = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $this->fieldname);
		$options   = [];

		if (!$this->contactId) {
			return $options;
		}

		$helper = new SismosappointmentHelper(['contactId' => $this->contactId,'moduleId' => $this->moduleId]);

		$period = $helper->getBookableTimeFrames();
		$intervalString = ((int) $this->timezoneOffset < 0) ? 'PT' . ((int) ($this->timezoneOffset) * -1) . 'M' : 'PT' . (int) ($this->timezoneOffset) . 'M';
		$interval = new \DateInterval($intervalString);
		
		foreach ($period as $group => $groupDate) {
			if (empty($groupDate)) {
				$date = Factory::getDate($group, 'UTC');
				if ((int) $this->timezoneOffset > 0) {
					$date->sub($interval);
				} else {
					$date->add($interval);
				}
				$group = $date->format('Y-m-d');
				$this->addOption($group, ['value' => '', 'group' => $group, 'disabled' => true, 'disabledDay' => ' disabled']);
				continue;
			}
			foreach ($groupDate as $option) {
				$date = Factory::getDate($option, 'UTC');
				if ((int) $this->timezoneOffset > 0) {
					$date->sub($interval);
				} else {
					$date->add($interval);
				}
				$value = $date->format('Y-m-d H:i:s');
				$text = $value;
				$group = $date->format('Y-m-d');
				$this->addOption($text, ['value' => $value, 'group' => $group, 'disabledDay' => '']);
			}
		}
		// }

		foreach ($this->element->xpath('option') as $option) {
			$value = (string) $option['value'];
			$text  = trim((string) $option) != '' ? trim((string) $option) : $value;
			$group = (string) $option['group'];

			$disabledDay = (string) $option['disabledDay'];

			$disabled = (string) $option['disabled'];
			$disabled = ($disabled === 'true' || $disabled === 'disabled' || $disabled === '1');
			$disabled = $disabled || ($this->readonly && $value != $this->value);
			$checked = false;

			$selected = false;

			$tmp = [
					'value'    => $value,
					'text'     => Text::alt($text, $fieldname),
					'disable'  => $disabled,
					'class'    => (string) $option['class'],
					'selected' => ($checked || $selected),
					'checked'  => ($checked || $selected),
					'group'    => $group,
					'disabledDay' => $disabledDay,
			];

			// Set some event handler attributes. But really, should be using unobtrusive js.
			$tmp['onclick']  = (string) $option['onclick'];
			$tmp['onchange'] = (string) $option['onchange'];

			// Add the option object to the result set.
			$options[] = (object) $tmp;
		}

		reset($options);

		return $options;
	}

	/**
	 * Method to get the data to be passed to the layout for rendering.
	 *
	 * @return  array
	 *
	 * @since 1.0.5
	 */
	protected function getLayoutData()
	{
		$data = parent::getLayoutData();

		if ($this->timezoneOffset != '') {
			$utcInfo = ((int) $this->timezoneOffset > 0) ? '-' : '+';
			$utcInfoTZOffset = ((float) $this->timezoneOffset < 0) ? ((float) $this->timezoneOffset * - 1) : (float) $this->timezoneOffset;
			$utcInfo .= \gmdate('H:i', floor((float) ($utcInfoTZOffset * 60)));
		} else {
			$utcInfo = \date('P');
		}

		$data['utcInfo'] = $utcInfo;

		$duration = \gmdate('H:i', floor((float) ($this->duration * 60)));

		$data['durationInfo'] = $duration;

		return $data;
	}
}
