<?php
/**
 * @package     SISMOSAppointments
 * @subpackage	Sismos.Zoom
 *
 * @author     Martina Scholz <martina@simplysmart-it.de>
 * @copyright  (C) 2023 Martina Scholz, SimplySmart-IT <https://simplysmart-it.de>
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       https://simplysmart-it.de
 */

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

use Joomla\Application\WebApplicationInterface;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\String\StringHelper;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\Event;
use Joomla\Event\EventInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\Http\Exception\UnexpectedResponseException;
use Joomla\OAuth2\Client;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Log\Log;

/**
 * SimplySmart-IT Appointment Zoom Integration.
 *
 * @package     SISMOSAppointments
 * @subpackage	Sismos.Zoom
 * @since       1.0.1
 */
class PlgSismosappointmentZoom extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Application object
	 *
	 * @var    CMSApplication
	 * @since  1.0.0
	 */
	protected $app;

	/**
	 * Database object
	 *
	 * The database is injected by parent constructor
	 *
	 * @var    DatabaseDriver|\JDatabaseDriver
	 * @since  1.0.0
	 */
	protected $db;

	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 *
	 * @var    boolean
	 * @since  1.0
	 */
	protected $autoloadLanguage = true;

	 /**
	 * The http factory
	 *
	 * @var    HttpFactory
	 * @since  1.0.0
	 */
	private $httpFactory;

	/**
	 * Constructor
	 *
	 * @param   DispatcherInterface  &$subject  The object to observe
	 * @param   array                $config    An optional associative array of configuration settings.
	 *                                          Recognized key values include 'name', 'group', 'params', 'language'
	 *                                         (this list is not meant to be comprehensive).
	 *
	 * @since   1.0.0
	 */
	public function __construct(&$subject, $config = [])
	{
		parent::__construct($subject, $config);

		$this->app = Factory::getApplication();

		// Define the logger.
		Log::addLogger(['text_file' => 'plg_sismosappointment_zoom.php'], Log::ALL, ['plg_sismosappointment_zoom']);
	}

	/**
	 * @inheritDoc
	 *
	 * @return string[]
	 *
	 * @since 4.1.0
	 *
	 * @throws Exception
	 */
	public static function getSubscribedEvents(): array
	{
		// TODO map own events
		$app = Factory::getApplication();

		$mapping  = [];
		
		// Only allowed in the backend
		if ($app->isClient('administrator')) {
			$mapping['onSismosGenToken'] = 'generateToken';
		} else {
			$mapping['onCallbackIntegration'] = 'callbackIntegration';
			$mapping['onSismosMeetingActivate'] = 'createMeeting';
		}

		return $mapping;
	}

	/**
	 * Catch url for callback
	 * Internal processor for all error handlers
	 *
	 * @param   EventInterface  $event The onCallbackIntegrationEvent event.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 *
	 * @look public function onAfterInitialise()
	 */
	public function callbackIntegration(Event $event)
	{
		/** @var \Joomla\CMS\Application\CMSApplication $app */
		[$queryParts, $app] = $event->getArguments();

		$plugin = PluginHelper::getPlugin('sismosappointment', 'zoom');

		foreach ($queryParts as $parts) {
			if (StringHelper::strpos($parts, 'code=') !== false) {
				$token = str_replace('code=', '', $parts);
				$this->params->set('code', $token);
				$this->Authenticate();
				break;
			}
		}
		$url = Uri::root() . 'administrator/index.php?option=com_plugins&task=plugin.edit&extension_id=' . $plugin->id;

		$app->redirect($url, (int) 303);
	}

	/**
	 * Generate a token for Zoom oAuth.
	 * This method acts on table save, when a token doesn't already exist or a reset is required.
	 *
	 * @param   EventInterface  $event The onExtensionBeforeSave event.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function generateToken(EventInterface $event): void
	{
		/** @var Extension $table */
		[$extension] = $event->getArguments();

		if ($extension['type'] !== 'plugin') {
			return;
		}

		if ($extension['element'] !== 'zoom') {
			return;
		}

		if (!array_key_exists('clientid', $extension['params']) || !$extension['params']['clientid']) {
			return;
		}

		if (!array_key_exists('clientsecret', $extension['params']) || !$extension['params']['clientsecret']) {
			return;
		}

		$this->Authenticate();
	}
	

	/**
	 * Standard routine method for the get request routine.
	 *
	 * @since 1.0.0
	 * @throws Exception
	 */
	protected function Authenticate()
	{

		$redirect = Uri::root() . 'oauth/zoom';
		$options = [
			'redirecturi'     => $redirect,
			'clientid'        => $this->params->get('clientid', 1),
			'clientsecret'    => $this->params->get('clientsecret', ''),
			'tokenurl'        => 'https://zoom.us/oauth/token',
			'authurl'        => 'https://zoom.us/oauth/authorize',
		];

		$input = null;
		$code = false;

		$token = $this->params->get('token', '');
		$token = ($token) ? ArrayHelper::fromObject($token) : '';
		if ($token && array_key_exists('refresh_token', $token) && $token['refresh_token']) {
			$options['userefresh'] = true;
		} elseif (!$this->params->get('code', '')) {
			$options['sendheaders']	= true;
		} else {
			$input    = $this->app->getInput();
			$input->set('code', $this->params->get('code', ''));
			$code = $this->params->get('code', false);
		}

		$http     = HttpFactory::getHttp($options);

		$client = new Client($options, $http, $input, $this->app);

		if (array_key_exists('userefresh', $options) && $options['userefresh']) {
			// Workaround @see https://github.com/joomla-framework/oauth2/pull/20
			// cannot be lowercase @see libraries/vendor/joomla/oauth2/src/Client.php line 410
			$response = $this->refreshToken($client, $http, $token['refresh_token']);
		} else {
			// Workaround @see https://github.com/joomla-framework/oauth2/pull/20
			// cannot be lowercase @see libraries/vendor/joomla/oauth2/src/Client.php line 117
			$response = $this->clientAuthenticate($client, $http, $code);
		}

		if ($response instanceof UnexpectedResponseException) {
			Factory::getApplication()->enqueueMessage('PLG_SISMOSAPPOINTMENT_AUTH_ERROR', 'error');
			return;
		}

		if ($response) {
			$this->saveToken($response);
		}
	}

	private function saveToken($response)
	{
		$plugin = PluginHelper::getPlugin('sismosappointment', 'zoom');
		$this->params->set('token', $response);
		$this->params->set('code', '');
		$params = json_encode($this->params, JSON_UNESCAPED_SLASHES);
		try {
			$query = $this->db->getQuery(true);
			
			$query->update($this->db->quoteName('#__extensions'))
				->set($this->db->quoteName('params') . ' = :params')
				->where($this->db->quoteName('extension_id') . ' = :extid')
				->bind(':params', $params, ParameterType::STRING)
				->bind(':extid', $plugin->id, ParameterType::INTEGER);

			$this->db->setQuery($query);

			$this->db->execute();
		} catch (\Exception $e) {
			$this->log(Text::sprintf('PLG_SISMOSAPPOINTMENT_ZOOM_LOG_SAVETOKEN_ERROR', $e->getMessage() . ' -Response: ' . (is_array($response) ? print_r($response, true) : $response)), Log::ERROR);
			Factory::getApplication()->enque->setError($e->getMessage());
		}
	}

	/**
	 * Get zoom url for meeting
	 *
	 * @param   Event  $event  The event object
	 *
	 * @return  void
	 *
	 * @since   1.0.1
	 *
	 */
	public function createMeeting(Event $event)
	{
		$entry = $event->getArgument('entry');
		$user = (array_key_exists('igv_user', $entry) && $entry['igv_user']) ? $entry['igv_user'] : 'me';
		$url = 'https://api.zoom.us/v2/users/' . $user . '/meetings';
		$token = $this->params->get('token', '');
		if (!is_object($token) || !isset($token->access_token) || !$token->access_token) {
			$this->log('Meeting Url not available - no access token found', Log::WARNING);
			$event->addArgument('meeting_url', false);
			return;
		}

		$token = ArrayHelper::fromObject($token);

		$startDateTime = Factory::getDate($entry['appointment'], 'UTC');
		$start = $startDateTime->format('Y-m-d') . 'T' . $startDateTime->format('H:i:s');

		$data = [
			"agenda" 			=> $entry['vc_desc'] . ' - '. Factory::getApplication()->get('sitename'),
			"default_password"	=> false,
			"duration" 			=> (int) $entry['duration'],
			"start_time" 		=> $start,
			"timezone"			=> "UTC",
			"topic" 			=> $entry['vc_title'] . ' - '. Factory::getApplication()->get('sitename'),
			"type" 				=>  2,
		];

		$data = json_encode($data);

		$options = [
			'authmethod'	  => 'bearer',
			'accesstoken'	  => $token,
			'clientid'        => $this->params->get('clientid', 1),
			'clientsecret'    => $this->params->get('clientsecret', ''),
			'tokenurl'        => 'https://zoom.us/oauth/authorize',
			'authurl'         => 'https://zoom.us/oauth/authorize',
		];

		$http     = (new HttpFactory)->getHttp($options);
		$client = new Client($options, $http, null, null);

		$client->setToken($token);

		$token = $client->getToken();

		// Workaround @see https://github.com/joomla-framework/oauth2/pull/20
		// cannot be lowercase @see libraries/vendor/joomla/oauth2/src/Client.php line 410
		if (array_key_exists('expires_in', $token) && $token['created'] + $token['expires_in'] < time() + 20) {
			$this->log('----- Refresh Zoom OAuth Token started -----', Log::INFO);
			$refreshOptions = [
				'redirecturi'     => Uri::root() . 'oauth/zoom',
				'clientid'        => $this->params->get('clientid', 1),
				'clientsecret'    => $this->params->get('clientsecret', ''),
				'tokenurl'        => 'https://zoom.us/oauth/token',
				'authurl'        => 'https://zoom.us/oauth/authorize',
				'userefresh'	  => true,
			];
			$refreshHttp     = HttpFactory::getHttp($refreshOptions);
			$refreshClient = new Client($refreshOptions, $refreshHttp, null, $this->app);
			$token = $this->refreshToken($refreshClient, $refreshHttp, $token['refresh_token']);
			if ($token instanceof UnexpectedResponseException) {
				Factory::getApplication()->enqueueMessage('PLG_SISMOSAPPOINTMENT_AUTH_ERROR', 'error');
				$this->log('Meeting Url not available - refreshing access token not possible', Log::ERROR);
				$this->_sendErrorMessage('Error creating Zoom Meeting Url', sprintf('Error creating Zoom Meeting for user_id %s - refreshing access token not possible.', $entry['contact_id']));
				$event->addArgument('meeting_url', false);
				return;
			}
			
			$this->saveToken($token);
			$this->log('----- Refresh Zoom OAuth Token end -----', Log::INFO);
			
			$client->setToken($token);
		}

		$headers = [
			"accept" => "application/json",
			"Content-Type" => "application/json",
			// cannot be lowercase @see libraries/src/Http/Transport/CurlTransport.php line 87
			// @see https://github.com/joomla-framework/oauth2/pull/20
		];
		
		$this->log('----- Create Meeting Zoom started -----', Log::INFO);
		
		try {
			$response = $client->query($url, $data, $headers, 'post');
		} catch (Exception $e) {
			$this->log(sprintf('Error creating Zoom Meeting for user_id %s: Error: %s', $entry['contact_id'], $e->getMessage()), Log::ERROR);
			$this->_sendErrorMessage('Error creating Zoom Meeting Url', sprintf('Error creating Zoom Meeting for user_id %s: Error: %s', $entry['contact_id'], $e->getMessage()));
			$event->addArgument('meeting_url', false);
			return;
		}		

		$this->log(sprintf('Create Meeting Zoom response code %s', $response->code), Log::INFO);

		$body = json_decode($response->body, true);
		$meeting = (is_array($body) && array_key_exists('join_url', $body)) ? $body['join_url'] : false;
		
		$this->log('----- Create Meeting Zoom end -----', Log::INFO);

		$event->addArgument('meeting_url', $meeting);
	}

	/**
	 * Get the access token or redirect to the authentication URL.
	 *
	 * @param   Client                    $client   use this Joomla\OAuth2\Client since bugs are fixed
	 * @param   Joomla\Http\Http          $http     The HTTP client object to use in sending HTTP requests.
	 * @param	string			          $code     authorize code
	 *
	 * @return  array|boolean  The access token or false on failure
	 *
	 * @since   1.0.0
	 * @throws  UnexpectedResponseException
	 * @throws  \RuntimeException
	 * @see libraries/vendor/joomla/oauth2/src/Client.php use class Joomla\OAuth2\Client when bug with code param is fixed
	 */
	protected function clientAuthenticate($client, $http, $code = false)
	{
		if ($code) {
			$data = [
				'grant_type'    => 'authorization_code',
				'redirect_uri'  => $client->getOption('redirecturi'),
				'client_id'     => $client->getOption('clientid'),
				'client_secret' => $client->getOption('clientsecret'),
				'code'			=> $code,
			];

			$response = $http->post($client->getOption('tokenurl'), $data);

			if (!($response->code >= 200 && $response->code < 400)) {
				$this->log(sprintf(
					'Error code %s received requesting access token: %s.',
					$response->code,
					$response->body
				), Log::ERROR);
				throw new UnexpectedResponseException(
					$response,
					sprintf(
						'Error code %s received requesting access token: %s.',
						$response->code,
						$response->body
					)
				);
			}

			if ($this->isJsonResponse($response)) {
				$token = array_merge(json_decode((string) $response->body, true), ['created' => time()]);
			} else {
				parse_str((string) $response->body, $token);
				$token = array_merge($token, ['created' => time()]);
			}

			$client->setToken($token);

			return $token;
		}

		if ($client->getOption('sendheaders')) {
			$app = Factory::getApplication();

			if (!($app instanceof WebApplicationInterface)) {
				$this->log(\sprintf('A "%s" implementation is required to process authentication.', WebApplicationInterface::class), Log::ERROR);
				throw new \RuntimeException(
					\sprintf('A "%s" implementation is required to process authentication.', WebApplicationInterface::class)
				);
			}

			$app->redirect($client->createUrl());
		}

		return false;
	}

	/**
	 * Refresh the access token instance.
	 *
	 * @param   Client              $client   use this Joomla\OAuth2\Client since bugs are fixed
	 * @param   Joomla\Http\Http    $http     The HTTP client object to use in sending HTTP requests.
	 * @param   string              $token    The refresh token
	 *
	 * @return  array  The new access token
	 *
	 * @since   1.0.0
	 * @throws  UnexpectedResponseException
	 * @throws  \RuntimeException
	 *
	 * @see libraries/vendor/joomla/oauth2/src/Client.php
	 * TODO use class Joomla\OAuth2\Client when bug with headers fixed
	 */
	protected function refreshToken($client, $http, $token = null)
	{

		if (!$token) {
			$token = $client->getToken();

			if (!array_key_exists('refresh_token', $token)) {
				$this->log('No refresh token is available.', Log::ERROR);
				throw new \RuntimeException('No refresh token is available.');
			}

			$token = $token['refresh_token'];
		}

		$data = [
			'grant_type'    => 'refresh_token',
			'refresh_token' => $token,
			'client_id'     => $client->getOption('clientid'),
			'client_secret' => $client->getOption('clientsecret'),
		];

		$response = $http->post($client->getOption('tokenurl'), $data);

		if (!($response->code >= 200 || $response->code < 400)) {
			$this->log(sprintf('Error code %s received refreshing token: %s.', $response->code, $response->body), Log::ERROR);
			throw new UnexpectedResponseException(
				$response,
				sprintf(
					'Error code %s received refreshing token: %s.',
					$response->code,
					$response->body
				)
			);
		}

		if ($this->isJsonResponse($response)) {
			$token = array_merge(json_decode((string) $response->body, true), ['created' => time()]);
		} else {
			parse_str((string) $response->body, $token);
			$token = array_merge($token, ['created' => time()]);
		}

		$client->setToken($token);

		return $token;
	}

	/**
	 * Tests if given response contains JSON header
	 *
	 * @param   \Joomla\Http\Response  $response  The response object
	 *
	 * @return  bool
	 *
	 * @since   1.0.0
	 * @see https://github.com/joomla-framework/oauth2/pull/20
	 */
	private function isJsonResponse(\Joomla\Http\Response $response): bool
	{
		foreach ($response->getHeader('Content-Type') as $value) {
			if (strpos($value, 'application/json') !== false) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Method to send an error message to Admin.
	 *
	 * @param   string    $subject   The subject for the message.
	 * @param   string	  $message   The message text
	 *
	 * @return  boolean  True on success sending the email, false on failure.
	 *
	 * @since   1.0.1
	 */
	private function _sendErrorMessage($subject, $message)
	{

		// Messaging to admin on Error

		// Push a notification to the site's super users, send email to user fails so the below message goes out
		/** @var MessageModel $messageModel */
		$messageModel = $this->app->bootComponent('com_messages')->getMVCFactory()->createModel('Message', 'Administrator');

		$messageModel->notifySuperUsers(
			$subject,
			$message
		);
	}

	/**
	 * Log helper function
	 *
	 * @return	string
	 */
	public function log($msg, $type)
	{
		if ($this->params->get('log_on', 1)) {
			Log::add($msg, $type, 'plg_sismosappointment_zoom');
		}
	}
}
